from typing import Any, Dict, List, cast

import dataiku
import dataikuapi
import requests
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger

# Use cast to tell mypy that the return value will be a list of dicts
languages = cast(List[Dict[str, str]], dataiku_api.webapp_config.get("user_profile_languages", []))

SUPPORTED_LANGUAGES = [{"key": item["from"], "value": item["to"]} for item in languages]


def format_feedback_choices(choices: List[Any]):
    choices_final = []
    for choice in choices:
        try:
            choices_final.append(str(choice))
        except Exception as e:
            logger.warn("choice can't be parsed to str")
    return choices_final


def get_user(headers):
    try:
        auth_info = dataiku.api_client().get_auth_info_from_browser_headers(headers)
        user = auth_info["authIdentifier"]
    except (dataikuapi.utils.DataikuException, requests.exceptions.HTTPError) as e:
        logger.exception(f"Exception occurred: {str(e)}")
        user = "user_not_found"
    return user


def get_current_user_profile(headers, user_profile_sql_manager):
    user = get_user(headers)
    if user == "user_not_found":
        return None
    profile = user_profile_sql_manager.get_user_profile(user)
    return profile


def resolve_webapp_param(param_name: str, default_value: Any, advanced_mode_enabled: bool=False)->Any:
    default_precision = "(Default webapp value)."
    resolved_value: Any = default_value
    if advanced_mode_enabled:
        resolved_value = dataiku_api.webapp_config.get(param_name, default_value)
        if param_name in ["llm_temperature", "title_llm_temperature", "decision_llm_temperature"]:
            if (resolved_value is None) or (not isinstance(resolved_value, (float, int))): # Don't use if 'not resolved_value' as the 'param_name' can be '0'
                # Specific return for the unit-tests
                logger.info(f"The resolved '{param_name}' is: '{default_value}' {default_precision if resolved_value == default_value else '.'}")
                return default_value
            if resolved_value < 0: 
                resolved_value = default_value
            default_precision = '(The default temperature of the LLM-Mesh connection is used).'
    logger.info(f"The resolved '{param_name}' is: '{resolved_value}' {default_precision if resolved_value == default_value else '.'}")
    return resolved_value