import base64
import imghdr
from io import BytesIO
from typing import Dict, Optional

import dataiku
from common.backend.constants import PREVIEW_IMAGE_LONGSIDE, PREVIEW_IMAGE_SHORT_SIDE
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger
from PIL import Image


def get_empty_image_base64() -> str:
    # This is a 1x1 transparent PNG encoded in base64
    return "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mP8/wcAAgAB/ivFZ8sAAAAASUVORK5CYII="


def b64encode_image_from_full_folder_id(file_path: str, full_folder_id: str) -> str:
    """
    Load an image located in a project - folder.
    handle the case where the folder is in another project
    """

    logger.debug(f"Encode an image from {full_folder_id} - {file_path}")

    b64_image = get_empty_image_base64() 
    if file_path and full_folder_id and "." in full_folder_id and "." in file_path:

        file_format = file_path.split(".")[-1]
        project_key = full_folder_id.split(".")[0]
        folder_id = full_folder_id.split(".")[-1]

        try:
            client = dataiku.api_client()
            project = client.get_project(project_key)
            folder = project.get_managed_folder(folder_id)

            with folder.get_file(file_path) as fd:
                http_resp = fd.raw
                if http_resp.status != 200:
                    logger.warn(f"Failed to load file from {full_folder_id} - {file_path} - got HTTP status : {http_resp.status}")
                    return b64_image

                file_content = http_resp.data

            base64_encoded = base64.b64encode(file_content)
            b64_image = base64_encoded.decode("utf-8")

        except Exception as e:
            logger.warn(f"An error occured while loading image : {e}")

        return f"data:image/{file_format};base64,{b64_image}"

    return b64_image


def b64encode_image_from_path(file_path: Optional[str] = None, folder_id: Optional[str] = None) -> str:
    """
    Encodes an image from a specified file path to a base64-encoded string.

    Parameters:
    - file_path (str): The path to the image file to be encoded. This path must point to a valid
    image file in the managed folder configured in the webapp.

    Returns:
    - str: A base64-encoded string representing the image. This string can be directly used in data URIs
        or for storage in text-based formats.
    """
    if not file_path:
        raise ValueError("file_path must be set.")

    try:
        config: Dict[str, str] = dataiku_api.webapp_config
        upload_folder: str = folder_id or config["upload_folder"]
        file_folder = dataiku.Folder(upload_folder)

        with file_folder.get_download_stream(file_path) as stream:
            file_content = stream.read()
        base64_encoded = base64.b64encode(file_content)
        img_b64 = base64_encoded.decode("utf-8")
        return img_b64
    except Exception as err:
        logger.exception(f"Error getting file {file_path} from upload folder, error: {err}")
        return get_empty_image_base64()


def detect_image_format(image_bytes):
    # Open image bytes as bytes stream
    byte_stream = BytesIO(image_bytes)
    # Use imghdr.what() method which takes a byte stream
    image_format = imghdr.what(byte_stream)

    return image_format


def get_image_bytes(image_path: str) -> Optional[bytes]:
    folder = dataiku_api.folder_handle
    # List all files in the folder
    files = folder.list_paths_in_partition()
    image_bytes = None
    if image_path in files or "/"+image_path in files:
        # Read the file as bytes
        with folder.get_download_stream(image_path) as stream:
            image_bytes = stream.read()
        logger.debug("Successfully retrieved the image as bytes.")
    else:
        logger.debug(f"File '{image_path}' not found in the folder.")
    return image_bytes


def resize_with_ratio(image):
    width, height = image.size
    scale = min(PREVIEW_IMAGE_SHORT_SIDE / width, PREVIEW_IMAGE_LONGSIDE / height)
    scale = min(1.0, scale)
    new_w = int(width * scale)
    new_h = int(height * scale)
    return image.resize((new_w, new_h), Image.LANCZOS)