from pathlib import Path
from typing import Any, Dict

from common.backend.models.base import MediaSummary, UploadFileError
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger

webapp_config: Dict[str, str] = dataiku_api.webapp_config

SUMMARY_TAG = "_extracted_text"


def handle_metadata_file_path(file_path: str, file_index: int = 1) -> str:
    folder = dataiku_api.folder_handle
    all_paths = folder.list_paths_in_partition()
    base_file_path = file_path.replace(".json", "")
    metadata_file_path = file_path
    while f"/{metadata_file_path}" in all_paths:
        logger.debug(f"File '{metadata_file_path}' already exists")
        metadata_file_path = f"{base_file_path}{file_index}.json"
        file_index += 1
    return metadata_file_path


def save_extracted_json(file_path: str, media_summary: MediaSummary) -> str:
    try:
        folder = dataiku_api.folder_handle
        fpath = Path(file_path)
        file_path = str(fpath.with_suffix(''))
        metadata_path = file_path.lower() + f"{SUMMARY_TAG}.json"
        # This avoids issue when 2 files have the same name but different extensions
        metadata_path = handle_metadata_file_path(metadata_path)
        folder.write_json(metadata_path, media_summary)
        return metadata_path
    except IOError as e:
        logger.exception(f"Error in save_extracted_json: {e}")
        raise Exception(UploadFileError.FILE_TOO_LARGE.value)

def read_extracted_json(file_path: str) -> MediaSummary:
    try:
        folder = dataiku_api.folder_handle
        media_summary_dict: MediaSummary = folder.read_json(file_path)
        return media_summary_dict
    except IOError as e:
        logger.exception(f"Error in read_extracted_json: {e}")
        raise Exception("Error in read_extracted_json")


def get_checked_config(variable_name: str) -> Any:
    value: Any = webapp_config.get(variable_name)
    if value is None:
        raise Exception(f"{variable_name} not set in webapp config")
    return value
