from typing import Optional

from common.llm_assist.logging import logger
from common.solutions.chains.generic_summary_chain import GenericSummaryChain
from dataikuapi.dss.llm import DSSLLMCompletionQuery


class TextExtractionSummaryChain(GenericSummaryChain):
    def __init__(self, extracted_text: str, original_file_name: str, language: Optional[str]):
        self._extracted_text = extracted_text
        self._original_file_name = original_file_name
        self._language = language

        self._completion = None

    @property
    def extracted_text(self) -> str:
        return self._extracted_text

    @property
    def original_file_name(self) -> str:
        return self._original_file_name

    @property
    def language(self) -> Optional[str]:
        return self._language

    def get_summary_completion(self) -> DSSLLMCompletionQuery:
        text_extraction_system_prompt = """
        # Role and Guidelines
        Your role is to read and summarize the text provided to you. This text is extracted from a document uploaded to a webapp.
        You must mention the contents of each section or chapter of the document if they exist.
        You should provide your answer in JSON format with the following keys
        - "summary": str: summary of the whole document provided to you
        - "topics": Array[str]: array of all the main topics discussed in the text provided.
        - "questions": Array[str]: array of 3 questions which can be asked based on the text provided to you. They should be fact based questions, not opinion based.
        The JSON object should have double quotes for keys and values. It is important to follow the JSON format strictly.
        Provide a JSON object only. Do not add any other explanations, decorations, or additional information beyond the JSON object.
        """
        text_extraction_user_prompt = f"""
        Summarize the following extracted document text
        # Extracted Text
        {self.extracted_text}
        # --- END OF EXTRACTED TEXT ---
        Your JSON object:
        """
        if self.language:
            text_extraction_system_prompt += f"""
        Your response should be in {self.language}
        """
        logger.debug(f"""
            Text Extraction Summary system Prompt: {text_extraction_system_prompt}
            Text Extraction Summary user Prompt: {text_extraction_user_prompt}
        """, log_conv_id=True)
        completion: DSSLLMCompletionQuery = self.init_completion()
        completion.with_message(text_extraction_system_prompt, role="system")
        completion.with_message(text_extraction_user_prompt, role="user")
        return completion
