import time
import unittest
from unittest.mock import patch

# Assuming the functions are part of a module named `user_profile_utils`
from common.backend.utils.user_profile_utils import (
    get_image_height,
    get_image_quality,
    get_image_width,
    get_nbr_images_to_generate,
    get_num_images_user_can_generate,
    set_nbr_images_to_generate,
    update_user_profile_generated_media_info,
)


class TestUserUtils(unittest.TestCase):
    @patch("common.backend.utils.user_profile_utils.dataiku_api")
    def test_get_num_images_user_can_generate(self, mock_dataiku_api):
        # Edge case: Empty user profile
        user_profile_empty = {}
        mock_dataiku_api.webapp_config = {"max_images_per_user_per_week": 10}
        result = get_num_images_user_can_generate(user_profile_empty)
        self.assertEqual(result, 10)

        # Edge case: Missing nested structure
        user_profile_missing_structure = {"media": {"image": {}}}
        result = get_num_images_user_can_generate(user_profile_missing_structure)
        self.assertEqual(result, 10)
        # Case : older than a week
        user_profile = {
            "media": {"image": {"nbr_images_to_generate": {"value": 10}}},
            "generated_media_info": {
                "image": {
                    "count": 5,
                    "first_generated_at": int(time.time()) - 604820,  # Older than a week
                }
            },
        }
        result = get_num_images_user_can_generate(user_profile)
        self.assertEqual(result, 10)
        # Case : exactly one week
        user_profile = {
            "media": {"image": {"nbr_images_to_generate": {"value": 10}}},
            "generated_media_info": {
                "image": {
                    "count": 5,
                    "first_generated_at": int(time.time()) - 604800,  # Equal to a week
                }
            },
        }
        result = get_num_images_user_can_generate(user_profile)
        self.assertEqual(result, 5)
        # Case : no images generated
        user_profile = {
            "media": {"image": {"nbr_images_to_generate": {"value": 10}}},
        }
        result = get_num_images_user_can_generate(user_profile)
        self.assertEqual(result, 10)
        # Case : max images generated
        user_profile = {
            "media": {
                "image": {
                    "nbr_images_to_generate": {"value": 10},
                }
            },
            "generated_media_info": {
                "image": {
                    "count": 10,
                    "first_generated_at": int(time.time()),
                }
            },
        }
        result = get_num_images_user_can_generate(user_profile)
        self.assertEqual(result, 0)

        # Case : some images generated
        user_profile = {
            "media": {
                "image": {
                    "nbr_images_to_generate": {"value": 10},
                }
            },
            "generated_media_info": {
                "image": {
                    "count": 8,
                    "first_generated_at": int(time.time()),
                }
            },
        }
        result = get_num_images_user_can_generate(user_profile)
        self.assertEqual(result, 2)

        mock_dataiku_api.webapp_config = {"max_images_per_user_per_week": 0}
        result = get_num_images_user_can_generate(user_profile_empty)
        self.assertEqual(result, 0)

    def test_get_image_height(self):
        # Edge case: Missing 'height'
        user_profile_missing_height = {"media": {"image": {}}}
        self.assertIsNone(get_image_height(user_profile_missing_height))

        # Case: Empty value 'height'
        user_profile_missing_height = {"media": {"image": {"height": {"value": ""}}}}
        self.assertIsNone(get_image_height(user_profile_missing_height))

        # Regular case
        user_profile_with_height = {"media": {"image": {"height": {"value": 1080}}}}
        self.assertEqual(get_image_height(user_profile_with_height), 1080)

    def test_get_image_width(self):
        user_profile_with_width = {"media": {"image": {"width": {"value": 1920}}}}
        self.assertEqual(get_image_width(user_profile_with_width), 1920)

        # Case: Empty value 'width'
        user_profile_missing_width = {"media": {"image": {"width": {"value": ""}}}}
        self.assertIsNone(get_image_width(user_profile_missing_width))

        # Edge case: 'width' not in profile
        user_profile_without_width = {"media": {"image": {}}}
        self.assertIsNone(get_image_width(user_profile_without_width))

    def test_get_image_quality(self):
        user_profile_with_quality = {"media": {"image": {"quality": {"value": "high"}}}}
        self.assertEqual(get_image_quality(user_profile_with_quality), "high")

        # Case: Empty value 'quality'
        user_profile_missing_quality = {"media": {"image": {"quality": {"value": ""}}}}
        self.assertIsNone(get_image_quality(user_profile_missing_quality))

        # Edge case: 'quality' not in profile
        user_profile_without_quality = {"media": {"image": {}}}
        self.assertIsNone(get_image_quality(user_profile_without_quality))

    def test_get_nbr_images_to_generate(self):
        user_profile_with_nbr_images = {"media": {"image": {"nbr_images_to_generate": {"value": 5}}}}
        self.assertEqual(get_nbr_images_to_generate(user_profile_with_nbr_images), 5)

        # Edge case: 'nbr_images_to_generate' not in profile
        user_profile_without_nbr_images = {"media": {"image": {}}}
        self.assertEqual(get_nbr_images_to_generate(user_profile_without_nbr_images), 1)

        # Case: 'nbr_images_to_generate' empty value
        user_profile_without_nbr_images = {"media": {"image": {"nbr_images_to_generate": {"value": ""}}}}
        self.assertEqual(get_nbr_images_to_generate(user_profile_without_nbr_images), 1)

    def test_set_nbr_images_to_generate(self):
        user_profile = {"media": {"image": {"nbr_images_to_generate": {"value": 5}}}}
        updated_profile = set_nbr_images_to_generate(user_profile, 10)
        self.assertEqual(updated_profile["media"]["image"]["nbr_images_to_generate"]["value"], 10)

        # Edge case: 'nbr_images_to_generate' key missing
        user_profile_without_field = {"media": {"image": {}}}
        self.assertEqual(set_nbr_images_to_generate(user_profile_without_field, 10), user_profile_without_field)

    @patch("time.time", return_value=1633036800)  # Mock current time to a specific timestamp
    def test_first_image_generation(self, mock_time):
        user_profile = {}
        nbr_images_generated = 5
        result = update_user_profile_generated_media_info(user_profile, nbr_images_generated)

        self.assertEqual(result["generated_media_info"]["image"]["count"], 5)
        self.assertEqual(result["generated_media_info"]["image"]["first_generated_at"], 1633036800)

    @patch("time.time", return_value=1633036800)  # Mock current time to a specific timestamp
    def test_time_has_passed(self, mock_time):
        user_profile = {
            "generated_media_info": {
                "image": {
                    "count": 10,
                    "first_generated_at": 1632432000,  # One week before the mocked time
                }
            }
        }
        nbr_images_generated = 3

        with patch("common.backend.utils.user_profile_utils.time_has_passed", return_value=True):
            result = update_user_profile_generated_media_info(user_profile, nbr_images_generated)

        self.assertEqual(result["generated_media_info"]["image"]["count"], 3)
        self.assertEqual(result["generated_media_info"]["image"]["first_generated_at"], 1633036800)

    @patch("time.time", return_value=1633036800)  # Mock current time to a specific timestamp
    def test_update_count_before_week(self, mock_time):
        first_generated_at_time = 1632936800  # Less than a week before the mocked time
        user_profile = {
            "generated_media_info": {
                "image": {
                    "count": 10,
                    "first_generated_at": first_generated_at_time,  # Less than a week before the mocked time
                }
            }
        }
        nbr_images_generated = 3

        with patch("common.backend.utils.user_profile_utils.time_has_passed", return_value=False):
            result = update_user_profile_generated_media_info(user_profile, nbr_images_generated)

        self.assertEqual(result["generated_media_info"]["image"]["count"], 13)
        self.assertEqual(result["generated_media_info"]["image"]["first_generated_at"], first_generated_at_time)
