from functools import lru_cache
from typing import Dict, List, Tuple

import dataiku
from common.backend.models.base import (
    MediaSummary,
)
from common.llm_assist.logging import logger
from dataikuapi.dss.project import DSSProject
from portal.backend.constants import AUG_RAG_TYPE, RA_MODEL_REWRITE_MODE
from portal.backend.utils.permissions_utils import user_has_access_to_project


def get_project_agents(current_project: DSSProject) -> List[Dict[str, str]]:
    return [
        {
            "value": current_project.project_key + ":" + llm.get("id"),
            "label": llm.get("friendlyName"),
            "description": "",  # description will be filled manually by admin
        }
        for llm in current_project.list_llms()
        if llm.get("id").startswith("agent")
    ]


def list_selected_agents(payload):
    selected_agents = payload["rootModel"].get("agents_ids", [])
    selected_projects = payload["rootModel"].get("projects_keys", [])
    agents_map = map_agents_id_name(selected_projects, with_project_key=True)
    return [{"value": agent, "label": agents_map.get(agent)} for agent in selected_agents]


@lru_cache(maxsize=128)
def _list_agents_by_project(selected_projects: Tuple[str, ...]) -> Dict[str, List[Dict[str, str]]]:
    client = dataiku.api_client()
    agents_by_project: Dict[str, List[Dict[str, str]]] = {}
    if not selected_projects:
        return agents_by_project
    for project_key in selected_projects:
        project_ob: DSSProject = client.get_project(project_key)
        agents = get_project_agents(project_ob)
        if agents:
            agents_by_project[project_key] = agents
    return agents_by_project


def list_agents_by_project(selected_projects) -> Dict[str, List[Dict[str, str]]]:
    return _list_agents_by_project(tuple(selected_projects))


@lru_cache(maxsize=128)
def _map_agents_id_name(selected_projects, with_project_key: bool = False) -> Dict[str, str]:
    agents_by_project = list_agents_by_project(selected_projects)
    agents_map = {}
    for project_key, agents in agents_by_project.items():
        for agent in agents:
            agents_map[agent["value"]] = agent["label"] if not with_project_key else f"[{project_key}] {agent['label']}"
    return agents_map


def map_agents_id_name(selected_projects, with_project_key: bool = False) -> Dict[str, str]:
    return _map_agents_id_name(selected_projects=tuple(selected_projects), with_project_key=with_project_key)

@lru_cache(maxsize=128)
def _list_augmented_llms_by_project(selected_projects: Tuple[str, ...]):
    aug_llm_by_project: Dict[str, List[Dict[str, str]]] = {}
    client = dataiku.api_client()
    for project in selected_projects:
        project_ob = client.get_project(project)
        aug_llms = [llm for llm in project_ob.list_llms() if llm.get("type") == AUG_RAG_TYPE]
        smart_mode_models = []
        if aug_llms:
            for model in aug_llms:
                rag_model_obj = project_ob.get_saved_model(model.get("savedModelSmartId"))
                rag_model_settings = rag_model_obj.get_settings().get_raw()
                active_version = rag_model_obj.get_active_version()
                if active_version:
                    active_model = next(
                        el
                        for el in rag_model_settings.get("inlineVersions")
                        if el.get("versionId") == active_version.get("id")
                    )
                    strategy = (
                        active_model.get("ragllmSettings", {}).get("searchInputStrategySettings", {}).get("strategy")
                    )
                    if strategy == RA_MODEL_REWRITE_MODE:
                        smart_mode_models.append(
                            {
                                "value": project + ":" + model.get("id"),
                                "label": model.get("friendlyName"),
                                "description": "",  # description will be filled manually by admin
                            }
                        )
            if smart_mode_models:
                aug_llm_by_project[project] = smart_mode_models
    return aug_llm_by_project

def list_augmented_llms_by_project(selected_projects):
    return _list_augmented_llms_by_project(tuple(selected_projects))


@lru_cache(maxsize=128)
def _map_aug_llms_id_name(selected_projects, with_project_key: bool = False) -> Dict[str, str]:
    llms_by_project = list_augmented_llms_by_project(selected_projects)
    aug_llms_map = {}
    for project_key, llms in llms_by_project.items():
        for llm in llms:
            aug_llms_map[llm["value"]] = llm["label"] if not with_project_key else f"[{project_key}] {llm['label']}"
    return aug_llms_map


def map_aug_llms_id_name(selected_projects, with_project_key: bool = False) -> Dict[str, str]:
    return _map_aug_llms_id_name(selected_projects=tuple(selected_projects), with_project_key=with_project_key)


def filter_agents_per_user(user: str, agents: List[str]) -> List[str]:
    accessible_agents = []
    projects_permissions: Dict[str, bool] = {}
    for agent in agents:
        project_key = agent.split(":")[0]
        if project_key in projects_permissions:
            if projects_permissions[project_key]:
                accessible_agents.append(agent)
            continue
        if user_has_access_to_project(user, project_key):
            accessible_agents.append(agent)
            projects_permissions[project_key] = True
        else:
            projects_permissions[project_key] = False
    return accessible_agents


def add_agent_uploads(
    previous_agents_uploads: Dict[str, Dict[str, MediaSummary]],
    agent_uploads: List[MediaSummary],
    agent_id: str,
):
    logger.debug(f"Adding agent uploads to previous uploads")
    # TODO should we keep preview + extracted_text or remove
    for upload in agent_uploads:
        file_name = upload.get("original_file_name", "")
        if file_name in previous_agents_uploads:
            previous_agents_uploads[file_name][agent_id] = upload
        else:
            previous_agents_uploads[file_name] = {agent_id: upload}
    return previous_agents_uploads
