import json
from typing import Any, Dict, List, Optional

from common.backend.models.base import LLMCompletionSettings, LlmHistory
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.llm_utils import handle_prompt_media_explanation
from common.llm_assist.logging import logger
from common.solutions.chains.generic_decision_chain import GenericDecisionJSONChain
from portal.backend.models import AgentDetails, AgentQuery


class AgentQueryBuilder(GenericDecisionJSONChain):
    def __init__(
        self,
        chat_history: List[LlmHistory],
        agents: List[AgentDetails],
        completion_settings: LLMCompletionSettings,
        chat_has_media: bool = False,
        user_profile: Optional[Dict[str, Any]] = None,
    ) -> None:
        self.agents = agents
        include_user_profile = dataiku_api.webapp_config.get("include_user_profile_in_prompt", False)
        self._completion_settings = completion_settings
        self._chat_history = chat_history

        profile_examples = (
            [
                """
            # USER PROFILE:
            {
                "country": {
                    "value": "USA",
                    "description": "country"
                }
            }
            # --- END OF USER PROFILE ---
        """,
                """
            # USER PROFILE:
            {
                "department": {
                    "value": "IT",
                    "description": "the department of the user"
                }
            }
            # --- END OF USER PROFILE ---
    """,
                """
            # USER PROFILE:
            {
                "group": {
                    "value": "Data Scientists",
                    "description": "the LDAP group of the user"
                }
            }
            # --- END OF USER PROFILE ---
    """,
            ]
            if include_user_profile
            else ["", "", ""]
        )
        profile_example1_mention1 = "in the USA" if include_user_profile else ""
        profile_example1_mention2 = (
            "In addition the user profile indicates that the user is in the USA" if include_user_profile else ""
        )
        examples_template = """
        ### Example Scenarios:
            #### Example 1:
            {profile_example_1}
            [{{"role": "user", "content": "Hello"}},
            {{"role": "assistant", "content": "Hi there! How can I help you?"}},
            {{"role": "user", "content": "What are the vacation policies and health benefits?"}},]
            EXPECTED OUTPUT: {{
                "queries": [
                    {{"agent_id": "answer:vaca:djk", "query": "What are the vacation policies at my company?"}},
                    {{"agent_id": "agent:healhy", "query": "What health benefits are available at my company?"}}
                ],
                "justification": "The user asked about topics covered by the 'answer:vaca:djk' and 'agent:healhy'."
            }}}}

            #### Example 2:
            {profile_example_2}
            [{{"role": "user", "content": "Good morning"}},
            {{"role": "assistant", "content": "Good morning! How may I assist you today?"}}،
            {{"role": "user", "content": ""Just checking in, thanks!"}}]
            EXPECTED OUTPUT: {{
                "queries": null,
                "justification": "The user's input is a terminal statement and does not require additional queries."
            }}

            #### Example 3:
            {profile_example_3}
            [{{"role": "user", "content": "Hello"}},
            {{"role": "assistant", "content": "Hello! How may I assist you today?"}}،
            {{"role": "user", "content": "Tell me about our company values and recent projects."}}]
            EXPECTED OUTPUT: {{{{
                "queries": [
                    {{"agent_id": "agent:val:ss", "query": "What are the company's values?"}},
                    {{"agent_id": "answers:ddw:dnkew", "query": "What are the recent projects undertaken by the company?"}}
                ],
                "justification": "The user's query relates to 'agent:val:ss' and 'answers:ddw:dnkew', warranting queries to these agents."
            }}
    """
        examples = examples_template.format(
            profile_example_1=profile_examples[0],
            profile_example_2=profile_examples[1],
            profile_example_3=profile_examples[2],
            profile_example1_mention1=profile_example1_mention1,
            profile_example1_mention2=profile_example1_mention2,
        )
        agents_str = json.dumps(self.agents)

        user_profile_mention = ", user profile" if include_user_profile else ""
        system_prompt = f"""
        You are a search query builder for different external Agents. Your role is to analyze the chat history{user_profile_mention} and the latest user input. 
        You decide whether additional information retrieval is necessary and, if required, craft one query per relevant agent based on the available agent descriptions.

        # Agents Overview
        Below is the detailed information about the available agents:
        ## Agents IDs and their Descriptions:
        {agents_str}
        ## -- End of detailed information --

        # Your Task
        Analyze the chat history and the available agents and their descriptions. If the chat suggests a need for more information on topics covered by the agents, produce a JSON object with two fields: 'queries' and 'justification'. 
        The 'queries' field should be a list of dictionary objects, each containing 'agent_id' and 'query'. 
        agent_id is selected only among existing agents ids provided in the section "## Agents IDs and their Descriptions:"
        If no queries are needed, 'queries' should be null, and 'justification' should explain why no queries are necessary. 
        Provide the JSON object only. Do not add any other explanations, decorations, or additional information.

        # Decision Criteria:
        - Relevant Inquiry: Generate a search query for each agent that aligns with a specific need implied by the user input and chat history. In case of doubt, craft a query to the closest matching agent or multiple queries to the closest matching agents if needed.
        - Response Protocol: For any user input that does not require further information from the agents, the 'queries' should be null. This is to maintain focus and relevance.
        - Irrelevant or Terminal Input: For brief greetings, closure statements, or unrelated questions, respond with 'queries' as null and provide a justification.
        {examples}
        """
        user_profile_section = (
            f"""
        # USER PROFILE:
        The following is the user’s profile, which provides context about the user:
        {user_profile}
        # --- END OF USER PROFILE ---        
        """
            if include_user_profile
            else ""
        )

        system_prompt = handle_prompt_media_explanation(system_prompt=system_prompt, has_media=chat_has_media)
        user_profile_mention2 = (
            "Consider information provided by the user profile and, if meaningful, take it into account while crafting the search query."
            if include_user_profile
            else ""
        )
        user_prompt = """
        Evaluate the chat history to determine if search queries should be generated for the agents. Generate the response as a JSON object with two fields: 'queries' and 'justification'. 
        Create search queries only if there is a need for more information on topics explicitly covered by the agents. 
        {user_profile_mention}
        If no queries are needed, 'queries' should be null and 'justification' should explain why no queries are necessary. 
        Provide the JSON object only. Do not add any other explanations, decorations, or additional information beyond the JSON object.
        {user_profile_section}
        """.format(user_profile_section=user_profile_section, user_profile_mention=user_profile_mention2)

        self._prompt = f"""
        {system_prompt}

        {user_prompt}
        """

    @property
    def prompt(self):
        return self._prompt

    @property
    def completion_settings(self) -> LLMCompletionSettings:
        return self._completion_settings

    @property
    def chat_history(self) -> List[LlmHistory]:
        return self._chat_history

    @property
    def chain_purpose(self):
        return "Agent Query Builder Chain"

    def verified_json_output(self, json_output) -> Dict[str, Any]:
        if (
            not json_output.get("queries")
            or not isinstance(json_output["queries"], list)
            or not any(json_output["queries"])
        ):
            return {
                "queries": None,
                "justification": json_output.get("justification"),
            }
        queries: List[AgentQuery] = json_output.get("queries")
        valid_id_queries = []
        invalid_agent_id_generated = False

        # Collect the valid agent IDs for easy lookup
        valid_agent_ids = {agent.get("agent_id") for agent in self.agents}

        # Filter out invalid agent IDs
        for q in queries:
            agent_id = q.get("agent_id")
            if agent_id in valid_agent_ids:
                valid_id_queries.append(q)
            else:
                logger.warn(f"Invalid agent id '{agent_id}' was filtered out.")
                invalid_agent_id_generated = True

        # If no valid queries remain after filtering, return queries=None
        if not valid_id_queries:
            return {
                "queries": None,
                "justification": json_output.get("justification"),
                "invalid_agent_id_generated": invalid_agent_id_generated,
            }

        return {
            "queries": valid_id_queries,
            "justification": json_output.get("justification"),
            "invalid_agent_id_generated": invalid_agent_id_generated,
        }

    def get_decision_json_pattern(self) -> str:
        return r"{(?:[^{}]|{(?:[^{}]|{[^{}]*})*})*}"
