from typing import Dict, List

import dataiku
from portal.backend.utils.agents_utils import (
    list_agents_by_project,
    list_augmented_llms_by_project,
    list_selected_agents,
    map_aug_llms_id_name,
)
from portal.backend.utils.answers_utils import get_answers_info


def list_answers(payload):
    selected_projects = payload["rootModel"].get("projects_keys", [])
    answers = get_answers_info(selected_projects)
    return {
        "choices": [
            {
                "value": webapp_id,
                "label": f"[{webapp['project_key']}] {webapp['webapp_name']}",
                "description": webapp["webapp_description"],
            }
            for webapp_id, webapp in answers.items()
        ]
    }

def list_selected_augmented_llms(payload):
    selected_aug_llms = payload["rootModel"].get("augmented_llms", [])
    selected_projects = payload["rootModel"].get("projects_keys", [])
    aug_llms_map = map_aug_llms_id_name(selected_projects, with_project_key=True)
    return [{"value": llm, "label": aug_llms_map.get(llm)} for llm in selected_aug_llms]

def list_augmented_llms(payload):
    selected_projects = payload["rootModel"].get("projects_keys", [])
    aug_llms_by_project = list_augmented_llms_by_project(selected_projects)
    choices = []
    for project_key, llms in aug_llms_by_project.items():
        for llm in llms:
            choices.append(
                {
                    "value": llm["value"],
                    "label": f"[{project_key}] {llm['label']}",
                    "description": llm["description"],
                }
            )
    return {"choices": choices}


def list_projects():
    projects = dataiku.api_client().list_projects()
    return {
        "choices": [
            {"value": project["projectKey"], "label": f"[{project['projectKey']}] {project['name']}"}
            for project in projects
        ]
    }


def list_agents(payload):
    selected_projects = payload["rootModel"].get("projects_keys", [])
    agents_by_project = list_agents_by_project(selected_projects)
    choices = []
    for project_key, agents in agents_by_project.items():
        for agent in agents:
            choices.append(
                {
                    "value": agent["value"],
                    "label": f"[{project_key}] {agent['label']}",
                    "description": agent["description"],
                }
            )
    return {"choices": choices}


def list_user_languages(payload) -> Dict[str, List]:
    defined_user_languages = payload["rootModel"].get("user_profile_languages", [])
    return {
        "choices": [
            {"value": lang.get("to", "English"), "label": lang.get("to", "English")} for lang in defined_user_languages
        ]
    }


def do(payload, config, plugin_config, inputs):  # noqa: PLR0911
    parameter_name = payload["parameterName"]
    client = dataiku.api_client()
    current_project = client.get_default_project()
    if parameter_name == "llm_id":
        return {
            "choices": [
                {"value": llm.get("id"), "label": llm.get("friendlyName")}
                for llm in current_project.list_llms()
                if llm.get("type") != "RETRIEVAL_AUGMENTED"
            ]
        }
    elif parameter_name == "image_generation_llm_id":
        return {
            "choices": [
                {"value": llm.get("id"), "label": llm.get("friendlyName")}
                for llm in current_project.list_llms(purpose="IMAGE_GENERATION")
                if llm.get("type") != "RETRIEVAL_AUGMENTED"
            ]
        }
    elif parameter_name == "projects_keys":
        return list_projects()
    elif parameter_name == "answers_ids":
        return list_answers(payload)
    elif parameter_name == "augmented_llms":
        return list_augmented_llms(payload)
    elif parameter_name == "agents_ids":
        return list_agents(payload)
    elif parameter_name == "agent_id":
        return {"choices": list_selected_agents(payload)}
    elif parameter_name == "augmented_llm":
        return {"choices": list_selected_augmented_llms(payload)}
    elif parameter_name == "default_user_language":
        return list_user_languages(payload)
    else:
        return {
            "choices": [
                {
                    "value": "wrong",
                    "label": f"Problem getting the name of the parameter.",
                }
            ]
        }
