import unittest
from unittest.mock import MagicMock, patch

from common.backend.db.sql.oracle import OracleParams, OracleSQLManager


def get_debug_calls(mock_logger: MagicMock) -> list[str]:
    return [call[0][0] for call in mock_logger.debug.call_args_list]


class TestOracle(unittest.TestCase):
    @patch("common.backend.db.sql.oracle.logger")
    def test_oracle_table_creation(self, mock_logger):
        mock_logger.debug = MagicMock()
        columns = ["col1", "col2", "query", "history", "metadata"]

        class MockDataset:
            def get_location_info(self):
                return {"locationInfoType": "SQL", "info": {"table": "test_table", "schema": "test_schema"}}

        mocked_dataset = MockDataset()
        params = OracleParams(
            dataset_name="test_dataset",
            executor=MagicMock(),
            dataset=mocked_dataset,
            columns=columns,
        )
        om = OracleSQLManager(params)
        om.setup_oracle_table()
        debug_calls = get_debug_calls(mock_logger)
        create_table_debug = debug_calls[0]
        print("create_table_debug", create_table_debug)
        self.assertIn("CREATE TABLE", create_table_debug)
        self.assertIn(""""col1" nvarchar2(2000)""", create_table_debug)
        self.assertEqual(create_table_debug.count("nvarchar2"), 2)
        self.assertIn(""""history" NCLOB,""", create_table_debug)
        self.assertEqual(create_table_debug.count("NCLOB"), 3)

    @patch("common.backend.db.sql.oracle.logger")
    def test_oracle_insert(self, mock_logger):
        mock_logger.debug = MagicMock()
        columns = ["col1", "metadata"]
        values = [["dummy", "dummy_metadata"]]
        OracleSQLManager.insert_statement(cols=columns, values=values)
        debug_calls = get_debug_calls(mock_logger)
        insert_debug = debug_calls[0]
        self.assertIn("dummy", insert_debug)
        self.assertIn("dummy_metadata", insert_debug)
        self.assertIn("TO_CLOB('dummy_metadata')", insert_debug)
        self.assertEqual(insert_debug.count("TO_CLOB"), 1)

    @patch("common.backend.db.sql.oracle.logger")
    def test_oracle_update(self, mock_logger):
        mock_logger.debug = MagicMock()
        mock_set = {"col1": "dummy", "metadata": "dummy_metadata"}
        OracleSQLManager.update_statement(sets=mock_set)
        debug_calls = get_debug_calls(mock_logger)
        update_debug = debug_calls[0]
        self.assertIn("dummy", update_debug)
        self.assertIn("dummy_metadata", update_debug)
        self.assertEqual(update_debug.count("TO_CLOB"), 1)
        self.assertEqual(update_debug.count("="), 2)
        self.assertEqual(update_debug.count(","), 1)
