from common.backend.models.base import Feedback, QuestionData
from common.llm_assist.log_decorators import log_http_request
from common.llm_assist.logging import logger
from flask import Blueprint, g, request
from portal.backend.db.conversations import conversation_sql_manager

from ..utils import before_request, return_ok

conversation_blueprint = Blueprint("conversation", __name__, url_prefix="/conversation")


@conversation_blueprint.before_request
def before_conversation_request():
    before_request()


@conversation_blueprint.route("/conversations", methods=["GET"])
@log_http_request
def get_user_conversations():
    auth_identifier = g.get("authIdentifier")
    logger.debug(f"User {auth_identifier} is fetching conversations.")
    conversations = conversation_sql_manager.get_user_conversations(auth_identifier)
    return return_ok(conversations)


@conversation_blueprint.route("/conversations", methods=["DELETE"])
@log_http_request
def delete_all_user_conversations():
    auth_identifier = g.get("authIdentifier")
    conversation_sql_manager.delete_all_user_conversations(auth_identifier=auth_identifier)
    return return_ok()


@conversation_blueprint.route("/<string:id>", methods=["GET"])
@log_http_request
def get_user_conversation(id: str):
    auth_identifier = g.get("authIdentifier")
    conversation = conversation_sql_manager.get_conversation(auth_identifier, id, False)
    if conversation and conversation.get("data"):
        for data in conversation["data"]:
            generated_media = data.get("generated_media")
            images = generated_media["images"] if generated_media and generated_media.get("images") else None
            data["generated_images"] = images
            data["generated_media"] = None
    return return_ok(conversation)


@conversation_blueprint.route("/<string:id>", methods=["DELETE"])
@log_http_request
def delete_user_conversation(id: str):
    auth_identifier = g.get("authIdentifier")
    conversation_sql_manager.delete_user_conversation(auth_identifier=auth_identifier, conversation_id=id)
    return return_ok()


@conversation_blueprint.route("/<string:id>/history", methods=["DELETE"])
@log_http_request
def clear_conversation_history(id: str):
    auth_identifier = g.get("authIdentifier")
    conversation_sql_manager.clear_conversation_history(auth_identifier=auth_identifier, conversation_id=id)
    return return_ok()


@conversation_blueprint.route("/<string:id>/records/<int:order>/log_feedback", methods=["POST"])
@log_http_request
def log_feedback(id: str, order: int):
    """Saves feedback to cache and logs it to the db"""

    auth_identifier = g.get("authIdentifier")
    conversation_id = id
    record_order = order
    request_as_json: QuestionData = QuestionData(**request.get_json())
    feedback = request_as_json["feedback"]
    if feedback is None:
        return

    # Update feedback in cache
    conversation_sql_manager.update_feedback(
        auth_identifier=auth_identifier,
        conversation_id=conversation_id,
        message_id=record_order,
        feedback=Feedback(
            value=feedback["value"],
            message=feedback["message"],
            choice=feedback["choice"],
        ),
    )

    logger.debug(f"Successfully recorded feedback: {feedback}")

    return return_ok()
