from typing import Any, Dict, List, Optional

from pydantic import BaseModel, Field


class Tool(BaseModel):
    """Tool reference - actual metadata comes from DSS"""

    id: str


class Document(BaseModel):
    """Document metadata"""

    name: str
    active: bool = True
    delete_pending: bool = False


class AgentBase(BaseModel):
    """Base agent fields"""

    name: str
    description: str = ""
    system_prompt: str = ""
    project_key: str = ""
    kb_description: str = ""
    sample_questions: List[str] = Field(default_factory=list)
    short_example_queries: List[str] = Field(default_factory=list)
    llmid: str = ""
    embedding_llm_id: str = ""
    tool_ids: List[str] = Field(default_factory=list)


class AgentCreate(AgentBase):
    """Request model for creating an agent"""

    id: Optional[str] = None
    owner: Optional[str] = None


class AgentUpdate(BaseModel):
    """Request model for updating an agent - all fields optional"""

    name: Optional[str] = None
    description: Optional[str] = None
    system_prompt: Optional[str] = None
    kb_description: Optional[str] = None
    sample_questions: Optional[List[str]] = None
    llmid: Optional[str] = None
    embedding_llm_id: Optional[str] = None
    tool_ids: Optional[List[str]] = None
    documents: Optional[List[Dict]] = None
    indexing: Optional[Dict[str, Any]] = None
    published_at: Optional[str] = None
    publishing_status: Optional[str] = None
    publishing_job_id: Optional[str] = None


class AgentResponse(AgentBase):
    """Response model with all agent data"""

    id: str
    owner: str
    documents: List[Document] = Field(default_factory=list)
    created_at: str
    last_modified: str
    indexing: Optional[Dict[str, Any]] = None
    published_version: Optional[Dict[str, Any]] = None
    published_at: Optional[str] = None
    publishing_status: Optional[str] = None

    # These will be populated from DSS when needed
    tools: Optional[List[Dict[str, Any]]] = None  # Full tool info from DSS
