export interface TextCompletionModel {
  id: string
  name: string
}

export interface ConversationStarterExample {
  id: string
  label: string
  query: string
  enterpriseAgents: string[]
}

export interface EnterpriseAgentItem {
  id: string
  projectKey?: string
  type?: string
  name: string
  sourceAgentName?: string
  description?: string
  exampleQuestions?: string[]
  additionalInstructions?: string
  allowInsightsCreation?: boolean
  storiesWorkspace?: string
}

export type PromptAttributes = Partial<{ enabled: boolean }>

export interface Prompt {
  id: string
  name: string
  description: string
  prompt: string
  category: string
  tags: string[]
  isDefault: boolean
  createdAt: string
  updatedAt: string
  createdBy?: string
}

export interface PromptsLibrary {
  version: string
  prompts: Prompt[]
}

export const enum ExtractionMode {
  PagesScreenshots = 'pagesScreenshots',
  PagesText = 'pagesText',
}

export interface AdminConfig {
  version?: string

  required?: string[]

  agentHubLLM?: string
  agentHubOptionalInstructions?: string
  orchestrationMode?: 'tools' | 'manual'

  smartMode?: boolean
  permanentlyDeleteMessages?: boolean
  allowDisableAgents?: boolean
  logsLevel?: 'info' | 'debug'
  conversationStarterExamples?: ConversationStarterExample[]

  enableDocumentUpload?: boolean
  extractionMode?: ExtractionMode
  textExtractionType?: 'IGNORE' | 'OCR' | 'VLM_ANNOTATE'
  maxImagesInConversation?: number
  uploadManagedFolder?: string
  conversationVisionLLM?: string

  myAgentsEnabled?: boolean
  myAgentsTextCompletionModels?: TextCompletionModel[]
  myAgentsEmbeddingModel?: string
  myAgentsFsConnection?: string
  myAgentsFolder?: string
  myAgentsNumDocs?: number
  myAgentsManagedTools?: string[]
  myAgentsEnablePromptLibrary?: boolean

  enterpriseAgents?: EnterpriseAgentItem[]

  chartsGenerationMode?: 'auto' | 'manual' | 'disabled'
  chartsTextCompletionModel?: string
  chartsMaxArtifactsSize?: number

  promptsAttributesById?: Record<string, PromptAttributes>

  myAgentsEnableGroupsRestriction?: boolean
  myAgentsExcludedShareGroups?: string[]
}

export function isFieldRequired(config: AdminConfig | undefined, fieldName: string): boolean {
  return config?.required?.includes(fieldName) ?? false
}

export function getRequiredFields(config: AdminConfig | undefined): string[] {
  return config?.required ?? []
}

export function setFieldRequired(config: AdminConfig, fieldName: string, required: boolean): void {
  if (!config.required) {
    config.required = []
  }

  const index = config.required.indexOf(fieldName)

  if (required && index === -1) {
    config.required.push(fieldName)
  } else if (!required && index !== -1) {
    config.required.splice(index, 1)
  }
}

export function addRequiredFields(config: AdminConfig, fieldNames: string[]): void {
  if (!config.required) {
    config.required = []
  }

  for (const fieldName of fieldNames) {
    if (!config.required.includes(fieldName)) {
      config.required.push(fieldName)
    }
  }
}

export function removeRequiredFields(config: AdminConfig, fieldNames: string[]): void {
  if (!config.required) return

  config.required = config.required.filter((field) => !fieldNames.includes(field))
}

export function isPromptEnabledFromConfig(config: AdminConfig | undefined, id: string): boolean {
  return config?.promptsAttributesById?.[id]?.enabled === false ? false : true
}
