from dataiku.connector import Connector
from airtable import airtable_api, get_base_schema
from airtable_common import get_api_key_from_config, RecordsLimit


class AirtableConnector(Connector):
    """
    A connector for interacting with Airtable bases and tables.

    Attributes:
        base (str): The ID of the Airtable base to interact with.
        table (str): The ID or name of the table within the base to interact with.
        key (str): The PAT key for authenticating with the Airtable base.
        retrieve_id (str): Indicates whether to retrieve the record ID. Can be 'yes' or 'no'.
    """

    def __init__(self, config, plugin_config):
        """
        Initializes the AirtableConnector with the given configuration and plugin configuration.

        :param config: The configuration for the connector.
        :param plugin_config: The plugin configuration for the connector.
        """
        Connector.__init__(self, config, plugin_config)  # Pass the parameters to the base class

        self.base = self.config.get("base_select")
        if not self.base or self.base == "_dku_manual_select":
            self.base = self.config.get("base")

        self.table = self.config.get("table_select")
        if not self.table or self.table == "_dku_manual_select":
            self.table = self.config.get("table")

        self.key = get_api_key_from_config(config)

        self.retrieve_id = self.config.get("retrieve_id", "no")

        if self.base is None or self.table is None or self.key is None:
            raise ValueError("Missing parameters (Base ID, or Table name, or API key")

    def get_read_schema(self):
        return None

    def generate_rows(self, dataset_schema=None, dataset_partitioning=None,
                      partition_id=None, records_limit=-1):

        limit = RecordsLimit(records_limit=records_limit)
        looping = True
        offset = None
        params = {
            "pageSize": 100
        }
        try:
            schema = get_base_schema(self.base, self.key)
            table = next(t for t in schema.get("tables") if t.get("id") == self.table)
            table_fields = {f.get("name") for f in table.get("fields")}
        except:
            # This feature is not accessible with legacy keys
            table_fields=[]
        while looping:
            if offset is not None:
                params.update({'offset': offset})
            results = airtable_api(self.base, self.table, self.key, parameters=params)
            for record in results.get("records"):
                if self.retrieve_id == 'yes':
                    record["fields"]["id"] = record["id"]
                record_fields = record["fields"]
                for field in table_fields:
                    if record_fields.get(field) is None:
                        record_fields[field] = ""
                yield record_fields
                if limit.is_reached():
                    return
            offset = results.get("offset")
            looping = False if offset is None else True
