"""
This files contains kind of a "wrapper function" for Airtable API and utility functions.
"""

import json
import requests
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter


# Basic logging
def log(*args):
    for thing in args:
        if type(thing) is dict:
            thing = json.dumps(thing)
        print('Airtable plugin - %s' % thing)


# Session object for requests
http_session = requests.Session()
# Retry strategy (cf http://stackoverflow.com/a/35504626/4969056)
retries = Retry(total=3,
                backoff_factor=2)
http_session.mount('https://', HTTPAdapter(max_retries=retries))


def airtable_api(base, table, token, action='', parameters={}, method='get', data={}):
    """
    Helper function to make calls to Airtable REST API.
    :param base: The base ID of the Airtable base.
    :param table: The table ID/name in the Airtable base.
    :param token: The PAT token for the Airtable base.
    :param action: The action to perform on the table (e.g. 'records', 'views', etc.).
    :param parameters: Additional parameters for the API request.
    :param method: The HTTP method to use for the request (e.g. 'get', 'post', etc.).
    :param data: The data to send with the request (for 'post' and 'put' methods).
    :return: The response from the Airtable API.
    """

    url = "https://api.airtable.com/v0/%s/%s/%s" % (base, table, action)
    return _send_request(method, url, token, json.dumps(data), parameters, 10)

def get_base_schema(baseId, token):
    """
    Get shema of all tables fom specified base
    :param baseId: The base ID of the Airtable base.
    :param token: The PAT token for the Airtable base.
    https://airtable.com/developers/web/api/get-base-schema
    """
    url = 'https://api.airtable.com/v0/meta/bases/%s/tables' % baseId
    return _send_request('get', url, token)


def _send_request(method, url, token, data=None, params=None, timeout=10):
    headers = {
        'Content-type': 'application/json',
        'Accept-Encoding': 'gzip',
        'Authorization': 'Bearer %s' % token
    }
    req = http_session.request(method, url, headers=headers)
    if method == 'get':
        req = http_session.request(method, url, headers=headers, params=params, timeout=timeout)
    elif method == 'post':
        req = http_session.request(method, url, headers=headers, data=data, params=params, timeout=timeout)
    log('API call: %s' % req.url)
    if req.status_code < 300:
        return req.json()
    else:
        error_message = get_error_message(req)
        raise Exception(error_message) 

def safe_airtable_api(base, table, token, action='', parameters={}, method='get', data={}):
    try:
        response = airtable_api(base, table, token, action=action, parameters=parameters, method=method, data=data)
    except Exception as error:
        return None, "{}".format(error)
    return response, None


def get_error_message(response):
    error_message = 'API error ({}) : {}'.format(response.status_code, response.content)
    log(error_message)
    try:
        json_content = json.loads(response.content)
    except Exception:
        return error_message
    enriched_error = json_content.get("error", {})
    if isinstance(enriched_error, dict):
        enriched_error_message = enriched_error.get("message", enriched_error)
    else:
        enriched_error_message = enriched_error
    if enriched_error_message:
        error_message = "{} - (Error {})".format(enriched_error_message, response.status_code)
    return error_message
