import os, json, requests, uuid
import yaml

import dataiku
from dataiku.code_studio import CodeStudioBlock

import utils

class OpenCodeBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        """
        :param config: the dict of the configuration of the object
        :param plugin_config: contains the plugin settings
        """
        self.config = config
        self.plugin_config = plugin_config
        
    def build_spec(self, spec, env):
        """
        Install OpenCode.ai in a Code Studio.
        
        :param spec: the current state of the CodeStudio template image spec
        :param env: the build env

        :returns: the updated spec, i.e. a dict with a field 'dockerfile'
        """
        
        # Add opencode dockerfile
        dockerfile = spec.get("dockerfile", "")
        
        dockerfile += utils.install.dockerfile(
            oc_version=self.config.get("opencode_version", ""),
            add_vscode_ext=self.config.get("add_vscode_ext", True),
            vscode_ext_version=self.config.get("vscode_ext_version", "")
        )
        
        # Add OpenCode_README.md to Docker build env
        readme_md = utils.install.get_readme_file()
        readme_filepath = os.path.join(env["buildDir"], "OpenCode_README.md")

        with open(readme_filepath, "w") as f:
            f.write(readme_md)
        
        return {"dockerfile":dockerfile}

    def build_launch(self, spec, env):
        """
        Apply the block to the spec
        
        :param spec: the current state of the CodeStudio launch spec
        :param env: the launch env

        :returns: the updated spec, as a dict
        """
        return spec
    
    def postprocess_launch(self, spec, env): 
        """
        Apply the block to the spec
        
        This method does two things:
        a) It builds a configmap with opencode.json (containing DSS auth information and 
           context windows for all models available to user through LLM Mesh)
        b) It modifies the Code Studio deployment yaml, mounting the configmap as a volume
        
        :param spec: the current state of the CodeStudio creation spec
        :param env: the creation env

        :returns: the updated spec, as a dict
        """
        code_studio_yaml = [ y for y in yaml.safe_load_all(spec["yaml"]) ]
        
        # (a) Build the opencode.json configmap       
        client = dataiku.api_client()
        opencode_json = utils.opencode.build_opencode_json(client, self.config, self.plugin_config)
        
        ## Configmap attributes: namespace, configmap name and key
        namespace = env["containerConfig"].get("kubernetesNamespace", "default") # k8s namespace where Code Studio is deployed
        execution_id = env["executionId"] # Code Studio execution ID, used to ensure name is unique
        
        configmap_name = f"opencode-configmap-{execution_id}"
        configmap_key = "opencode.json"
        volume_name = "opencode-config-vol"
        volume_path = configmap_key
        volume_mount_path = "/home/dataiku/.config/opencode"
        
        ## Build configmap
        opencode_configmap = utils.k8s.create_configmap(
            name=configmap_name,
            namespace=namespace,
            key=configmap_key,
            value=json.dumps(opencode_json, indent=2)
        )
        code_studio_yaml.append(opencode_configmap)
        
        # (b) Mount the configmap to the Code Studios pod as a volume
        for y in code_studio_yaml:
            if y["kind"] == "Deployment":
                utils.k8s.add_volume_to_deployment(
                    deployment=y, 
                    volume={
                        "name": volume_name,
                        "configMap": {
                            "name": configmap_name,
                            "items": [
                                {
                                    "key": configmap_key,
                                    "path": volume_path
                                }
                            ]
                        }
                    }
                )
                utils.k8s.mount_volume_in_all_containers(deployment=y, volume_name=volume_name, volume_mount_path=volume_mount_path)

                v = utils.k8s.get_env_var_from_container_deployment(deployment=y, env_var_name="DKU_BASE_CERT")
                cert = v.get("value", "")
                if cert != "":
                    import base64
                    cert_decoded = base64.b64decode(cert[4:]).decode("utf-8")
                    assert cert_decoded.startswith("-----BEGIN CERTIFICATE-----"), "Invalid certificate"
                    
                    cert_configmap_name = f"cert-configmap-{execution_id}"
                    cert_configmap_key = "dss_server.crt"
                    cert_volume_name = "opencode-cert-vol"
                    cert_volume_path = cert_configmap_key
                    cert_volume_mount_path = "/home/dataiku/.config/dss_cert"
                    cert_configmap = utils.k8s.create_configmap(
                        name=cert_configmap_name,
                        namespace=namespace,
                        key=cert_configmap_key,
                        value=cert_decoded
                    )
                    code_studio_yaml.append(cert_configmap)

                    utils.k8s.add_volume_to_deployment(
                        deployment=y, 
                        volume={
                            "name": cert_volume_name,
                            "configMap": {
                                "name": cert_configmap_name,
                                "items": [
                                    {
                                        "key": cert_configmap_key,
                                        "path": cert_volume_path
                                    }
                                ]
                            }
                        }
                    )
                    utils.k8s.mount_volume_in_all_containers(deployment=y, volume_name=cert_volume_name, volume_mount_path=cert_volume_mount_path)
                    utils.k8s.add_env_var_with_value_to_deployment(
                        deployment=y,
                        env_var_name="NODE_EXTRA_CA_CERTS",
                        env_var_value=f"{cert_volume_mount_path}/{cert_configmap_key}"
                    )
        
        # Return updated YAML spec        
        spec["yaml"] = yaml.safe_dump_all(code_studio_yaml)
        return spec

    def build_creation(self, spec, env):
        """
        Apply the block to the spec
        
        :param spec: the current state of the CodeStudio launch spec
        :param env: the launch env

        :returns: the updated spec, as a dict
        """
        # Add AGENTS.md to code studio resources
        md_tmp_filename = "AGENTS.md-" + str(uuid.uuid4())
        md_tmp_filepath = os.path.join(env["tmpDir"], md_tmp_filename)
        agents_md = utils.install.get_md_file()
        with open(md_tmp_filepath, "w") as f:
            f.write(agents_md)
        spec.get("codeStudioResourcesFiles", []).append({"key": md_tmp_filename, "value": "AGENTS.md"})
        return spec