'use strict';

const llmApp = angular.module('llmConfig.module', []);

llmApp.constant('PLUGIN_PATHS', {
    DOCUMENT_QA: '/plugins/document-question-answering/resource/custom-ui-settings/',
});

llmApp.constant('SUPPORTED_CONNECTIONS', [
    'PostgreSQL',
    'Snowflake',
    'Redshift',
    'MS SQL Server',
    'BigQuery',
    'Databricks',
    'Oracle'
]);


// -------------  Parent Controller & Its helper directive --------------------


llmApp.controller('LLMConfigController', ['$scope', '$timeout', 'PythonService', 'DataikuAPI','SUPPORTED_CONNECTIONS', 'DescParamsService', '$q', '$window', function ($scope, $timeout, PythonService, DataikuAPI,SUPPORTED_CONNECTIONS, DescParamsService, $q, $window) {


    $scope.newConversationDataset = false;
    $scope.newLoggingDataset = false;
    $scope.isHistoryDatasetsExpanded = false
    $scope.supportedConnections = SUPPORTED_CONNECTIONS

    $scope.isConversationsDatasetExpanded = false;
    $scope.toggleExpand = function(propertyName) {
        $scope[propertyName] = !$scope[propertyName];
    };

    //State handling for the `datasetSelector/datasetConnectionSelector` directives
    $scope.toggleFlag = function (name) {
        if (name === "Conversation") {
            $scope.newConversationDataset = !$scope.newConversationDataset;
        } else if (name === "Logging") {
            $scope.newLoggingDataset = !$scope.newLoggingDataset;
        }
    };


    DataikuAPI.admin.folderEdit.listContents("LOCAL_STATIC")
        .then(function (response) {


            return {
                success: true,
                data: response.data,
                status: response.status
            };
        })
        .catch(function (error) {

            return {
                success: false,
                error: error,
                message: error.message || 'Failed to retrieve content'
            };
        });


    //These array of sections will help us to validate the form and its subsequent subforms.
    $scope.sections = [{
        id: 'llmConfig', displayName: 'LLM and Datasets', formName: 'llmConfigSubForm'
    }, {
        id: 'llmConfigSection', displayName: 'LLM Configuration', formName: 'llmConfigSectionSubForm'
    }, {
        id: 'conversationStore', displayName: 'Conversation Store', formName: 'conversationStoreSubForm'
    }, {id: 'documentUpload', displayName: 'Document Upload', formName: 'documentUploadSubForm'},
        {id: 'imageGeneration', displayName: 'Image Generation', formName: 'imageGenerationSubForm'},
    {
        id: 'retrievalMode', displayName: 'Retrieval method', formName: 'retrievalModeSubForm'
    }, {
        id: 'answersApiConfiguration', displayName: 'Answers API', formName: 'answersApiSubForm'
    }, {
        id: 'endUserConfiguration', displayName: 'Interface Customization', formName: 'endUserConfigSubForm'
    }, {id: 'userProfileSettings', displayName: 'User Profile', formName: 'userProfileSubForm'}, {
        id: 'webApplication', displayName: 'WebApplication', formName: 'webApplicationSubForm'
    }, {id: 'generalFeedback', displayName: 'General Feedback', formName: 'generalFeedbackSubForm'}];

    // This is where the cache clearing and data refreshing is orchestrated.
    // We broadcast a new, generic event that all data-displaying components
    //The reason it is residing in this controller is because this is the parent controller for all the directives.
    //It will allow us to broadcast the event to its children
    var datasetCreatedListener = $scope.$on('datasetCreated', function () {

        $scope.$broadcast('dataNeedsRefresh'); // BROADCAST DOWNWARDS

    });

    // Clean up listener on Parent scope destroy
    $scope.$on('$destroy', function () {
        datasetCreatedListener();
    });


    $scope.navigateToSectionAndError = function (sectionWithError) {

        $location.hash(sectionWithError.id);

        // 2. Use $timeout to wait for the DOM to update after ng-show takes effect
        $timeout(function () {
            // At this point, the section should be visible
            let targetElement = null;

            if (sectionWithError.firstErrorId) {
                targetElement = document.getElementById(sectionWithError.firstErrorId);
            }
            // Assuming your section divs look like this: <div id="section-llmConfig" ng-show="selectedSection === 'llmConfig'">
            const sectionContainerId = 'section-' + sectionWithError.id;
            const sectionContainer = document.getElementById(sectionContainerId);

            if (sectionContainer) {
                sectionContainer.scrollIntoView({behavior: 'smooth', block: 'start'});
            } else {

                // Absolute fallback: scroll to the top of the page or the main form.
                const mainFormElement = document.getElementsByName('configForm')[0];
                if (mainFormElement) {
                    mainFormElement.scrollIntoView({behavior: 'smooth', block: 'start'});
                } else {
                    window.scrollTo({top: 0, behavior: 'smooth'});
                }
            }

        }, 100);
    };

    $scope.$watch(function () {
        return $location.hash();
    }, function (newHash) {
        if (newHash) {
            // Remove the # symbol and update selectedSection
            $scope.selectedSection = newHash.replace('#', '');
        } else {
            // Default to llmConfig when no hash is present
            $scope.selectedSection = 'llmConfig';
        }
    });

    $scope.config = $scope.config || {};

    if (typeof $scope.callPythonDo === 'function') {
        //Register the callPython method that will allow us to call the script `params_helper.py`
        PythonService.registerCallPythonDoFunction($scope.callPythonDo);
    } else {

        // Fallback to a dummy to avoid breaking PythonService if it's called before a proper registration
        PythonService.registerCallPythonDoFunction(function () {

            return $q.reject("callPythonDo was not properly initialized in LLMConfigController for PythonService.");
        });
    }

    // Register the desc params once in the parent
    DescParamsService.registerDescParams($scope);
    $scope.descParams = DescParamsService.getDescParams();

    if ($scope.selectedSection === undefined) {
        $scope.selectedSection = 'llmConfig'; // Default section
    }

    $scope.showScrollToTop = false;

    // Function to scroll to top
    $scope.scrollToTop = function () {
        var controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
        if (controllerElement) {
            controllerElement.scrollIntoView({behavior: 'smooth'});
        } else {
            // Fallback to window scroll
            window.scrollTo({top: 0, behavior: 'smooth'});
        }
    };

    // Function to check scroll position and show/hide button
    function checkScrollPosition() {
        // Target the actual scrollable element
        var scrollableElement = document.querySelector('[ng-controller="CustomWebAppEditController"]');

        if (!scrollableElement) {

            return;
        }

        var scrollTop = scrollableElement.scrollTop;
        var elementHeight = scrollableElement.clientHeight;
        var scrollHeight = scrollableElement.scrollHeight;

        // Show button when user has scrolled down more than 300px
        var scrollPercent = (scrollTop + elementHeight) / scrollHeight;

        $scope.$apply(function () {
            $scope.showScrollToTop = scrollTop > 300 && scrollPercent > 0.3;
        });
    }

    // Add scroll event listener to the correct element
    var scrollableElement = document.querySelector('[ng-controller="CustomWebAppEditController"]');
    if (scrollableElement) {
        angular.element(scrollableElement).on('scroll', checkScrollPosition);

    } else {

    }

    // Clean up event listener when scope is destroyed
    $scope.$on('$destroy', function () {
        if (scrollableElement) {
            angular.element(scrollableElement).off('scroll', checkScrollPosition);
        }
    });


}])

llmApp.directive('loadingWatcher', function ($rootScope, $timeout) {
    return {
        restrict: 'A',
        link: function (scope, element, attrs) {
            function checkSpinner() {
                const spinner = document.getElementById('.qa_spinner');
                $rootScope.isLoading = !!spinner;

                if (spinner) {
                    $timeout(checkSpinner, 100); // keep checking until gone
                }
            }

            // Run once document is ready
            $timeout(checkSpinner, 0);
        }
    };
});

llmApp.directive('listenForSaveAndView', function () {
    return {
        restrict: 'A', link: function (scope) {
            function handleClick(event) {
                var target = event.target;

                while (target && target !== document) {
                    if (target.tagName === 'BUTTON' || target.tagName === 'A') {
                        var elementText = target.textContent.trim().toLowerCase();

                        if (elementText.includes('save and view webapp') || elementText.includes('save and view app') || elementText.includes('save & view') || elementText.includes('save')) {

                            // Temporarily stop the event
                            event.stopPropagation();
                            event.preventDefault();

                            // Apply to ensure we're in Angular context
                            scope.$apply(function () {
                                // 1. Mark the form as submitted to trigger validations
                                if (scope.configForm) {
                                    scope.configForm.$submitted = true;
                                }

                                // 2. Run validation logic
                                var isValid = validateForm(scope);

                                if (isValid) {

                                    // If valid, either:
                                    // A) Manually trigger the original action
                                    $timeout(function () {
                                        // Simulate click on the original element
                                        // but flag it to avoid our interceptor
                                        target.setAttribute('data-bypassed', 'true');
                                        target.click();
                                    });
                                } else {

                                    // Show error message
                                    scope.showValidationError = true;

                                    // Scroll to the LLMConfigController div when validation fails
                                    $timeout(function () {
                                        var controllerElement = document.querySelector('[ng-controller="LLMConfigController"]');
                                        if (controllerElement) {
                                            controllerElement.scrollIntoView({behavior: 'smooth'});
                                        }
                                    });
                                }
                            });

                            return false;
                        }
                    }
                    target = target.parentElement;
                }
            }

            // Function to validate the form
            function validateForm(scope) {
                scope.invalidSections = [];
                scope.showGlobalError = false;
                let isOverallFormInvalid = false;

                // Access your form object
                var form = scope.configForm;

                if (!form) {

                    return false;
                }

                angular.forEach(scope.sections, function (section) {
                    // Check if the sub-form for this section exists and is invalid
                    const sectionForm = scope.configForm[section.formName];
                    if (sectionForm && sectionForm.$invalid) {
                        scope.invalidSections.push(section);
                        isOverallFormInvalid = true;
                    }
                });

                if (isOverallFormInvalid) {
                    scope.showGlobalError = true;
                    return false; // Indicates validation failed
                }

                // Check if the form is valid
                if (form.$valid) {
                    // Reset form state after successful validation
                    form.$setPristine();
                    form.$setUntouched(); // Also reset touched state for better UX
                    return true;
                }

                return false;
            }

            // Check for our own bypassed clicks to avoid infinite loops
            function isBypassedClick(target) {
                return target.hasAttribute('data-bypassed');
            }

            function clickHandler(event) {
                var target = event.target;

                // Skip if this is our own bypassed click
                while (target && target !== document) {
                    if (isBypassedClick(target)) {
                        // Remove the bypass flag after handling
                        target.removeAttribute('data-bypassed');
                        return;
                    }
                    target = target.parentElement;
                }

                // Otherwise proceed with normal handling
                handleClick(event);
            }

            // Add global listener
            document.addEventListener('click', clickHandler, true);

            // Clean up
            scope.$on('$destroy', function () {
                document.removeEventListener('click', clickHandler, true);
            });

        }
    };
});