llmApp.factory('KnowledgeBankService', ["DataikuAPI", function (DataikuAPI) {
    const SUPPORTED_METADATA_DATATYPES = ["string"];
    const SOURCE_FILE_METADATA = "source_file";
    const DISPLAY_ONLY_MULTIMODAL_METADATAS = ["source_pages"];
    const REPLACED_METADATAS = ["dku_file_path"] // 'source_file' is displayed in the UI instead of 'dku_file_path' for UX purposes 
    const SUPPORTS_VECTOR_DB_FILTER = ["QDRANT_LOCAL", "CHROMA", "PINECONE"]
    
    const service = {
        /**
         * List Knowledge Banks
        */
       listKnowledgeBanks: function(projectKey){
        let choices = [];
        return DataikuAPI.retrievableknowledge.list(projectKey).then(function (response) {
                const knowledgeBanks = response.data
                knowledgeBanks.forEach(item => {
                    choices.push({
                        value: item.id,
                        label: item.name
                    });
                });
                return {choices: choices};
            })
       },
       /**
         * Get Knowledge Bank search type
         * Adaptation of the python function 'get_kb_search_type'
        */
       getKbSearchType: function(projectKey, knowledgeBankId){
        let choices = [
            {value: "similarity", label: "Similarity score only"},
            {value: "dss_similarity_score_threshold", label: "Similarity score with threshold (unbounded)"},
            {value: "similarity_score_threshold", label: "Similarity score with threshold (bounded within [0, 1) )"},
        ];
        return service.getKnowledgeBankSettings(projectKey, knowledgeBankId).then(
            function (kBConfig) {
                if (kBConfig.vectorStoreType === "AZURE_AI_SEARCH"){
                    choices.push(
                        {value: "hybrid", label: "Hybrid search"},
                        {value: "semantic_hybrid", label: "Semantic hybrid search"},
                    );
                } else {
                    choices.push({value: "mmr", label: "Improve diversity of documents"})
                };
                return {choices: choices}
            }
        )
       },
        /**
         * Get all the Knowledge Bank information needed in Answers. 
         * Adaptation of the python function 'get_knowledge_bank_info'
        */
        getKnowledgeBankInfo: function (knowledgeBankId, flowNodesData, knowledgeBankSettings) {
            const kbRecipeNode = service.findAntecedentRecipeNode(flowNodesData, knowledgeBankId)
            const embedding_recipe_name = kbRecipeNode.name
            const embedding_recipe_type = kbRecipeNode.recipeType;
            const recipePredecessors = kbRecipeNode.predecessors || [];
            let metadata_dataset_id = "";
            let documents_folder_id = "";

            for (const predecessorId of recipePredecessors) {
                const predNode = flowNodesData[predecessorId];
                if (!predNode) continue;

                if (predNode.nodeType === "LOCAL_DATASET") {
                metadata_dataset_id = flowNodesData[predecessorId].name; // each node.name from 'flowNodesData' is a public API identifier
                } else if (predNode.nodeType === "LOCAL_MANAGED_FOLDER") {
                documents_folder_id = flowNodesData[predecessorId].name; // each node.name from 'flowNodesData' is a public API identifier
                }
            }

            const metadata_schema = [];
            const metadataColumnsSchema = knowledgeBankSettings.metadataColumnsSchema || [];

            for (const metadataInfo of metadataColumnsSchema) {
                const metadataDatatype = metadataInfo.type;
                if (SUPPORTED_METADATA_DATATYPES.includes(metadataDatatype)) {
                metadata_schema.push(metadataInfo.name);
                }
            }

            const vector_store_type = knowledgeBankSettings.vectorStoreType;
            return {
                embedding_recipe_name,
                embedding_recipe_type,
                vector_store_type,
                metadata_dataset_id,
                documents_folder_id,
                metadata_schema,
            };
        },
        /**
         * Finds the recipe that is the antecedent of a flow fraph graph node (the 'successorId' function param).
         * Adaptation of the python function 'find_antecedent_recipe'
        */
        findAntecedentRecipeNode: function ( flowNodesData, successorId) {
            let antecedentNode = null;
            for (const [nodeId, node] of Object.entries(flowNodesData)) {
                const nodeSuccessors = node.successors || [];
                for (const successor of nodeSuccessors) {
                    if (flowNodesData[successor].name === successorId) {// each node.name from 'flowNodesData' is a public API identifier
                        antecedentNode = node;
                        break;
                    }
                }
                if (antecedentNode !== null){
                    break;
                }
            };
        return antecedentNode;
        },
        /**
         * Get Knowledge Bank settings
        */
        getKnowledgeBankSettings: function (projectKey, knowledgeBankId) {
            return DataikuAPI.retrievableknowledge.get(projectKey, knowledgeBankId).then(function (response) {
                const res = response.data
                return res;
            })
            .catch(function (error) {
                Logger.error("Error fetching the Knowledge Bank config:", error);
                return {};
            });
        },
        /**
         * Get Flow graph
        */
       getFlowGraph: function(projectKey){
        return DataikuAPI.flow.recipes.getGraph(projectKey)
       },

        /**
         * Fetch Knowledge Bank metadata 
        */
        getKnowledgeBankMetadata: function (projectKey, knowledgeBankId, parameterName, forMultiselect) {
            let choices = [];
            return service.getFlowGraph(projectKey)
            .then(function (response) {
                    return  response["data"]["serializedFilteredGraph"]["serializedGraph"]["nodes"]
                }
            )
            .then(function (flowNodesData){
                return service.getKnowledgeBankSettings(projectKey, knowledgeBankId).then(
                    function (kBConfig) {
                        const KnowledgeBankInfo = service.getKnowledgeBankInfo(knowledgeBankId, flowNodesData, kBConfig)
                        if (forMultiselect) {
                            /**
                             * Covers 'knowledge_sources_context_metadata', 'knowledge_sources_filters', 'knowledge_sources_displayed_metas'
                             */
                            if (KnowledgeBankInfo.embedding_recipe_type === "embed_documents"){
                                choices.push({
                                    value: SOURCE_FILE_METADATA,
                                    label: SOURCE_FILE_METADATA
                                });
                                if (parameterName !== "knowledge_sources_filters"){
                                    DISPLAY_ONLY_MULTIMODAL_METADATAS.forEach(item => {
                                        choices.push({
                                            value: item,
                                            label: item
                                        });
                                    });
                                }
                            }
                            if ( (parameterName === "knowledge_sources_filters") && (!SUPPORTS_VECTOR_DB_FILTER.includes(KnowledgeBankInfo.vector_store_type))) {
                                return {choices: []}
                            }
                            KnowledgeBankInfo.metadata_schema.forEach(item => {
                                if (!REPLACED_METADATAS.includes(item)) {
                                        choices.push({
                                            value: item,
                                            label: item
                                        });
                                    }
                            });
                            return {choices: choices}                            

                            }
                        
                        else {
                            /**
                             * Covers 'knowledge_source_url', 'knowledge_source_title', 'knowledge_source_thumbnail'
                             */
                            if (KnowledgeBankInfo.embedding_recipe_type === "embed_documents"){
                                // Addition of the 'SOURCE_FILE_METADATA' in context of multimodal
                                // Note: we could avoid adding 'SOURCE_FILE_METADATA' when the metadata role is not 'knowledge_source_title'
                                choices.push({
                                    value: SOURCE_FILE_METADATA,
                                    label: SOURCE_FILE_METADATA
                                });
                            }
                            KnowledgeBankInfo.metadata_schema.forEach(item => {
                                if (!REPLACED_METADATAS.includes(item)) {
                                        choices.push({
                                        value: item,
                                        label: item
                                    });
                                }
                            });
                            return {choices: choices}
                        }
                }
            )
        }
    )
        },
    };
  return service;
}]);