llmApp.factory("ParamsHelperService", [
    "DataikuAPI",
    "APIXHRService",
    "$q",
    "KnowledgeBankService",
    "Logger",
    function (DataikuAPI, APIXHRService, $q, KnowledgeBankService, Logger) {
        const CONNECTION_TYPES = {
            POSTGRES: "PostgreSQL",
            SNOWFLAKE: "Snowflake",
            REDSHIFT: "Redshift",
            SQLSERVER: "SQLServer",
            BIGQUERY: "BigQuery",
            DATABRICKS: "Databricks",
            ORACLE: "Oracle"
        };
        const service = {
            /**
             * Main entry point - mimics the Python do() function
             */
            do: function (payload, config, pluginConfig, inputs) {
                const parameterName = payload.parameterName;
                const projectKey = payload.projectKey; // Get projectKey from payload
                const MULTISELECT_KB_METADATA= ["knowledge_sources_context_metadata", "knowledge_sources_filters", "knowledge_sources_displayed_metas"];                
                switch (parameterName) {

                    /* 
                    Application LLMs
                    */
                    case "llm_id":
                    case "json_decision_llm_id":
                    case "title_llm_id":
                    case "image_generation_llm_id":
                        return service.listLlmsByConnectionsType(projectKey, payload.purpose, payload.typesToExclude)
                
                    /*
                    Application SQL connection
                    */
                    case "connection_selector":
                    case "sql_retrieval_connection":
                        return service.listConnectionsByType();
                    
                    /*
                    'Single' select parameters
                    */
                   case "knowledge_bank_id":
                    return KnowledgeBankService.listKnowledgeBanks(projectKey);
                   
                   case "knowledge_retrieval_search_type":
                    if ((config.retrieval_mode === "kb") && (config.knowledge_bank_id)){
                        return KnowledgeBankService.getKbSearchType(projectKey, config.knowledge_bank_id)
                    }

                   case "upload_folder":
                    return service.listProjectManagedFolders(projectKey);
                    
                    /*
                    Knowledge Bank metadata parameters (Single & Multiselect)
                    */
                    case "knowledge_source_url":
                    case "knowledge_source_title":
                    case "knowledge_source_thumbnail":
                    case "knowledge_sources_context_metadata":
                    case "knowledge_sources_filters": 
                    case "knowledge_sources_displayed_metas":
                        if ((config.retrieval_mode === "kb") && (config.knowledge_bank_id)){
                            let forMultiselect = false;
                            if (MULTISELECT_KB_METADATA.includes(parameterName)){
                                forMultiselect = true;
                            }
                            return KnowledgeBankService.getKnowledgeBankMetadata(projectKey, config.knowledge_bank_id, parameterName, forMultiselect)
                        }
                    
                    case "sql_retrieval_table_list":
                        return service.listDatasetsFromConnection(projectKey, config.sql_retrieval_connection)
                    case "columns_mappings":
                        return service.getDatasetColumnsMapping(projectKey, payload.selectedDatasets)
                    case "datasets_names":
                        return service.listDatasetsByConnectionTypes(projectKey)
                    case "create_dataset":
                        return service.createDataset(projectKey, payload.arguments.name, payload.arguments.connection)
                    
                    default:
                        return $q.resolve({
                            choices: [
                                {
                                    value: "wrong",
                                    label: `Problem handling the parameter with payload ${JSON.stringify(payload)}.`,
                                },
                            ],
                        });
                }
            },
            /**
             * List LLMs by connection type
             */
            listLlmsByConnectionsType: function (projectKey, purpose, typesToExclude) {
                purpose = purpose || "GENERIC_COMPLETION";

                return DataikuAPI.pretrainedModels
                    .listAvailableLLMs(projectKey, purpose)
                    .then(function (response) {
                        const llmByType = {};

                        // Handle new schema: response.data.identifiers
                        const allLlms =
                            response.data && response.data.identifiers
                                ? response.data.identifiers
                                : [];
                        allLlms.forEach(function (llm) {
                            const llmType = llm.type;

                            if (typesToExclude.indexOf(llmType) !== -1) {
                                return;
                            }

                            if (!llmByType[llmType]) {
                                llmByType[llmType] = [];
                            }

                            llmByType[llmType].push({
                                value: llm.id,
                                label: llm.friendlyName,
                                type: llmType,
                            });
                        });

                        return {choices: llmByType};
                    })
                    .catch(function (error) {
                        Logger.error("Error listing LLMs by connection type", error);
                        return {choices: {}};
                    });
            },

            /**
             * List connections by type
             */
            listConnectionsByType: function () {
                const connectionTypes = Object.values(CONNECTION_TYPES);
                const promises = connectionTypes.map(function (connectionType) {
                    return DataikuAPI.connections.getNames(connectionType)
                        .then(function (response) {
                            const connections = response.data || [];
                            return {
                                type: connectionType,
                                connections: connections
                            };
                        })
                        .catch(function (error) {
                            Logger.error("Error processing connection type " + connectionType, error);
                            return {
                                type: connectionType,
                                connections: []
                            };
                        });
                });

                return $q.all(promises).then(function (results) {
                    const connectionMap = {};

                    results.forEach(function (result) {
                        if (result.connections && result.connections.length > 0) {
                            // Initialize array for this connection type if not exists
                            if (!connectionMap[result.type]) {
                                connectionMap[result.type] = [];
                            }
                            // Add each connection name to the array
                            result.connections.forEach(function (connectionName) {
                                connectionMap[result.type].push(connectionName);
                            });
                        }
                    });

                    return {choices: connectionMap};
                });
            },
            /**
             * List datasets from a given connection
             * Adaptation of the python function 'list_datasets_from_conn'
             */
            listDatasetsFromConnection: function (projectKey, connectionName) {
                let connectionType = "";
                let dataset_choices = [];
                return DataikuAPI.datasets.list(projectKey)
                    .then(function (response) {

                        response.data.forEach(item => {
                            if (item.params.connection === connectionName){
                                dataset_choices.push({
                                    value: item.name,
                                    label: item.name
                                });
                                connectionType = item.type
                            }
                        });

                        return {
                            choices: {
                                [connectionType]: dataset_choices
                            }
                        };

                    })
                    .catch(function (error) {
                        Logger.error(`Error listing the project datasets on the connection ${connectionName}`, error);
                        return {choices: [{ value: "", label: "None" }]};
                    });
            },
            /**
             * List selected SQL retrieval dataset columns
             * Adaptation of the python function 'get_possible_joins_columns'
             */
            getDatasetColumnsMapping: function (projectKey, datasetNames) {
                let choices = [];

                const promises = datasetNames.map(datasetName => {
                    return DataikuAPI.datasets.get(projectKey, datasetName, projectKey)
                        .then(function (response) {
                            response.data.schema.columns.forEach(colData => {
                                choices.push({
                                    value: `${datasetName}.${colData.name}`,
                                    label: `${datasetName}.${colData.name}`
                                });
                            });
                        })
                        .catch(function (error) {
                            Logger.error(
                                `Error listing the columns from project dataset '${datasetName}'`,
                                error
                            );
                            return null;
                        });
                });

                return Promise.all(promises).then(() => {
                    return { choices: choices };
                });
            },
            /**
             * List datasets by connection types
             * Adaptation of the python function 'list_datasets_by_connection_type'
             */
            listDatasetsByConnectionTypes: function (projectKey,) {
                let connectionDatasets = {};
                const connectionTypes = Object.values(CONNECTION_TYPES);
                return DataikuAPI.datasets.list(projectKey)
                .then(function (response) {
                    response.data.forEach(item => {
                        const connType = item.type;
                        const datasetName = item.name;
                        if (connectionTypes.includes(connType)) {
                            (connectionDatasets[connType] ??= []).push(datasetName);
                        }
                    });
                    return { choices: connectionDatasets };
                }).catch(function (error) {
                        Logger.error(`Error listing datasets by connection types`, error);
                        return {choices: [{ value: "", label: "None" }]};
                    });                
            },
            createDataset: function(projectKey, datasetName, connectionParams){

                const managedDatasetCreationSettings = {
                    connectionId: connectionParams.connectionName,
                    partitioningOptionId: "NP",
                    typeOptionId: null,
                    inlineDataset: false,
                    preserveOrder: true,
                    zone: null,
                    markCreatedAsBuilt: true,
                    specificSettings: {
                        FormatOptionId: null,
                        overrideSQLCatalog: null,
                        overrideSQLSchema: null,
                    }
                };
                console.log(`Creating the dataset ${datasetName}`)

                return DataikuAPI.datasets.newManagedDataset(projectKey, datasetName, managedDatasetCreationSettings)
                .then(
                    function (){
                        return {
                            success: true,
                            dataset: {
                                id: datasetName,
                                datasetName: datasetName,
                                datasetDisplayName : datasetName,
                                datasetType: connectionParams.connectionType
                            },
                            message: `Dataset '${datasetName}' created successfully`
                        }
                    }
                ).catch(function (error) {
                        Logger.error("Error creating the project dataset", error);
                        return {choices: [{ value: "", label: "None" }]};
                    });
            },
            /**
             * List project 'Managed' folders
             */
            listProjectManagedFolders: function (projectKey) {
                return DataikuAPI.managedfolder.list(projectKey)
                    .then(function (response) {
                        const choices = [
                            {value: "", label: "None"}
                        ];
                        response.data.forEach(item => {
                            choices.push({
                                value: item.id,
                                label: item.name
                            });
                        });
                        return {choices: choices};
                    })
                    .catch(function (error) {
                        Logger.error("Error listing the project managed folders", error);
                        return {choices: [{ value: "", label: "None" }]};
                    });
            },
        };

        return service;
    },
]);
