import json
import sys

from answers.backend.db.conversations import conversations_sql_manager
from answers.backend.db.messages import messages_sql_manager
from answers.backend.db.sql.dataset_schemas import CONVERSATION_DATASET_CONF_ID
from answers.backend.db.user_profile import user_profile_sql_manager
from answers.backend.routes.api.api_checking import (
    check_request_schema,
    get_api_dataset_status,
    get_endpoint_availability,
)
from answers.solutions.service import LLM_Question_Answering
from common.backend.constants import LLM_API_ERROR
from common.backend.schemas.ask import AskRequestSchema, AskResponseDataSchema
from common.backend.services.ask import AskService
from common.llm_assist.log_decorators import log_http_request
from common.llm_assist.logging import logger
from flask import Blueprint, Response, jsonify, request, stream_with_context

ask_bp = Blueprint("ask", __name__)


def str_to_bool(value):
    """Convert query string value to boolean."""
    return value.lower() in ["true", "1", "yes", "on"]


@ask_bp.route("/ask", methods=["POST"])
@log_http_request
def process_query():
    availability_response, status_code  = get_endpoint_availability("ask")
    if status_code != 200:
        return availability_response
    enable_conversation, _ = get_api_dataset_status(CONVERSATION_DATASET_CONF_ID)
    accept_header = request.headers.get("Accept", "text/event-stream").lower()

    if accept_header not in ["text/event-stream", "application/json"]:
        return jsonify(error="Not Acceptable: Supported types are text/event-stream, application/json."), 406

    streaming = accept_header == "text/event-stream"

    # TODO validate auth ?
    # Load request data and validate using schema
    request_data = request.get_json()
    schema_check_response, satus_code = check_request_schema(request_data, AskRequestSchema())
    if satus_code != 200:
        return schema_check_response
    llm_qa = LLM_Question_Answering()

    if streaming:

        def stream_answer():
            try:
                for chunk in AskService.process_query_streaming(
                    request_data=request_data,
                    llm_qa=llm_qa,
                    user_profile_sql_manager=user_profile_sql_manager,
                    messages_sql_manager=messages_sql_manager,
                    conversations_sql_manager=conversations_sql_manager if enable_conversation else None,
                ):
                    yield f"event: {chunk.event}"
                    yield "\n"
                    yield f"data: {json.dumps(chunk.data)}"
                    yield "\n\n"
                    yield ": flush\n\n"
                    sys.stdout.flush()

            except Exception as e:
                logger.exception(f"Error processing query: {e}")
                yield f"event: completion-end"
                yield "\n"
                yield f"data: {json.dumps({'status': 'ko', 'message': LLM_API_ERROR})}"
                yield "\n\n"
                return

        return Response(
            stream_with_context(stream_answer()),
            mimetype="text/event-stream; charset=utf-8",
            headers={
                "Cache-Control": "no-cache",
                "X-Accel-Buffering": "no",
                "Connection": "keep-alive",
            },
        )
    else:
        try:
            processed_data = AskService.process_query(
                data=request_data,
                llm_qa=llm_qa,
                user_profile_sql_manager=user_profile_sql_manager,
                messages_sql_manager=messages_sql_manager,
                conversations_sql_manager=conversations_sql_manager if enable_conversation else None,
            )

            # Build and validate the response using the schema
            response_schema = AskResponseDataSchema()
            validated_response = response_schema.dump(processed_data)
            # Return the response
            return jsonify({"status": "ok", "data": validated_response})
        except Exception as e:
            logger.exception(f"Error processing query: {e}")
            return jsonify({"status": "ko", "message": LLM_API_ERROR}), 500
