from answers.backend.db.conversations import conversations_sql_manager
from answers.backend.routes.api.api_checking import check_get_params, get_endpoint_availability
from common.backend.models.base import APIConversationsResponse, APISingleConversationResponse
from common.backend.schemas.conversations import ConversationsResponseSchema, SingleConversationResponseSchema
from common.llm_assist.log_decorators import log_http_request
from common.llm_assist.logging import logger
from flask import Blueprint, g, request

from ..utils import before_request, extract_get_params, return_ko, return_ok, return_ok_message

conversations_blueprint = Blueprint("conversations", __name__, url_prefix="/conversations")

@conversations_blueprint.before_request
def before_conversation_request():
    before_request()


@conversations_blueprint.route("/", methods=["GET"])
@log_http_request
def get_all_user_conversations():
    auth_identifier = g.get("authIdentifier")
    availability_response, status_code = get_endpoint_availability("conversations") 
    if status_code != 200:
        return availability_response

    get_params_error = check_get_params("applicationId", "user")
    if get_params_error:
        return return_ko(get_params_error), 400
    
    application_id, user = extract_get_params("applicationId", "user") 

    logger.debug(f"Auth user '{auth_identifier}' is fetching all conversations for user '{user}' (application_id={application_id})")

    conversations: APIConversationsResponse = conversations_sql_manager.get_all_user_conversations(application_id, user, only_present=False)
    response_schema = ConversationsResponseSchema()
    validated_response = response_schema.dump(conversations)
    return return_ok(validated_response)


@conversations_blueprint.route("/", methods=["DELETE"])
@log_http_request
def delete_all_user_conversations():
    auth_identifier = g.get("authIdentifier")
    availability_response, status_code = get_endpoint_availability("conversations") 
    if status_code != 200:
        return availability_response


    get_params_error = check_get_params("applicationId", "user")
    if get_params_error:
        return return_ko(get_params_error), 400
    
    application_id, user = extract_get_params("applicationId", "user") 

    logger.debug(f"Auth user '{auth_identifier}' is deleting all conversations for user '{user}' (application_id={application_id})")

    conversations_sql_manager.delete_all_user_conversations(application_id, user)
    return return_ok_message("Conversations deleted successfully")


@conversations_blueprint.route("/<string:conversation_id>", methods=["GET", "DELETE"])
@log_http_request
def handle_conversation(conversation_id: str):
    auth_identifier = g.get("authIdentifier")
    availability_response, status_code = get_endpoint_availability("conversations") 
    if status_code != 200:
        return availability_response

    get_params_error = check_get_params("applicationId", "user")
    if get_params_error:
        return return_ko(get_params_error), 400
    
    application_id, user = extract_get_params("applicationId", "user")
        
    if request.method == "GET":
        logger.debug(f"Auth user '{auth_identifier}' is getting the conversation '{conversation_id}' for user '{user}' (application_id={application_id})")
        conversation_messages: APISingleConversationResponse = conversations_sql_manager.get_conversation_messages(application_id, user, conversation_id, only_present=False)
        if conversation_messages:
            response_schema = SingleConversationResponseSchema()
            validated_response = response_schema.dump(conversation_messages)
            return return_ok(validated_response)
        else:
            return_ko(f"conversation_id '{conversation_id}' not found")
    
    elif request.method == "DELETE":
        logger.debug(f"Auth user '{auth_identifier}' is deleting the conversation '{conversation_id}' for user '{user}' (application_id={application_id})")
        conversations_sql_manager.delete_conversation(application_id, user, conversation_id)
        return return_ok_message("Conversation deleted successfully")