import traceback
from typing import Dict, List, Optional, Union

from answers.backend.db.general_feedback import feedbacks_sql_manager
from answers.solutions.knowledge_bank import get_knowledge_bank_full_name
from common.backend.utils.dataiku_api import dataiku_api
from common.backend.utils.date_utils import get_string_date_interpreted_as_string
from common.llm_assist.log_decorators import log_http_request
from common.llm_assist.logging import logger
from flask import Blueprint, Response, g, request

from ..utils import before_request, return_ko, return_ok
from .api_response_handler import APIGeneralFeedbackRequestProcessor

general_feedback_blueprint = Blueprint(
    "general_feedback", __name__, url_prefix="/general_feedback")


@general_feedback_blueprint.before_request
def before_general_feedback_request():
    before_request()


@general_feedback_blueprint.route("/submit", methods=["POST"])
@log_http_request
def submit_general_feedback() -> Response:
    """
    Submits a general feedback from a user into feedbacks dataset

    Returns:
        Response: A Flask response object containing feedback submission status.
    """

    request_json: Dict[str, Union[str, List[str]]] = request.get_json()
    logger.debug(f"Payload from the front end is {request_json}")

    api_response_handler: APIGeneralFeedbackRequestProcessor = APIGeneralFeedbackRequestProcessor(
        api_response=request_json
    )

    auth_identifier = g.get("authIdentifier")
    message: str = api_response_handler.extract_message()
    knowledge_bank_id: Optional[str] = api_response_handler.extract_knowledge_bank_id(
    )

    knowledge_bank_full_id = get_knowledge_bank_full_name(
        knowledge_bank_id)

    # Get llm id
    config: Dict[str, str] = dataiku_api.webapp_config
    llm_id = config.get("llm_id", None)
    current_utc_timestamp = get_string_date_interpreted_as_string(sql_dialect=feedbacks_sql_manager.dialect)

    try:
        feedbacks_sql_manager.add_feedback(
            current_utc_timestamp, auth_identifier, message, knowledge_bank_full_id, llm_id)
    except Exception as e:
        logger.error(f"An error occurred: {e}")
        traceback.print_exc()  # Print the full traceback
        return return_ko(f"An error occurred: {e}") # type: ignore
    return return_ok(data={"status": "ok"}) # type: ignore
