from typing import Dict, Tuple, Union

from answers.solutions.knowledge_bank import get_knowledge_bank_name
from common.backend.models.base import RetrieverInfo, RetrieverMode
from common.backend.utils.dataiku_api import dataiku_api
from common.llm_assist.logging import logger


def get_retriever_info(config: Dict[str, str]) -> Union[RetrieverInfo, None]:
    retrieval_mode = config.get("retrieval_mode")
    logger.debug(f"Retrieval mode: {retrieval_mode}")
    if retrieval_mode == RetrieverMode.KB.value:
        activate_knowledge_bank: bool = config.get("activate_knowledge_bank", True)  # type: ignore
        kb_id = config.get("knowledge_bank_id", None)
        if kb_id is None or kb_id == "":
            return None
        kb_name = get_knowledge_bank_name(kb_id)
        kb_custom_name = config.get("knowledge_bank_custom_name")
        return {
            "id": kb_id,
            "name": kb_name,
            "alias": kb_custom_name,
            "activated": activate_knowledge_bank,
            "type": RetrieverMode.KB.value,
        }
    elif retrieval_mode == RetrieverMode.DB.value:
        table_list = config.get("sql_retrieval_table_list", None)
        if table_list is None or table_list == []:
            return None
        table_list = ", ".join(table_list)
        # TODO: should this list of datasets have a different key?
        # TODO: should suggested joins be here too?
        return {
            "name": table_list,
            "id": table_list,
            "alias": config.get("sql_retrieval_custom_name", ""),
            "activated": True,
            "type": RetrieverMode.DB.value,
        }
    else:
        return None


def get_tool_type_and_name() -> Tuple[str, str]:
    """
    Retrieve the retrieval type (kb or db) and 
    the tool name from the configuration
    """
    config = dataiku_api.webapp_config
    retrieval_mode = config.get("retrieval_mode")

    tool_type = ""
    tool_name = ""

    if retrieval_mode == RetrieverMode.KB.value:
        tool_type = RetrieverMode.KB.value
        kb_id = config.get("knowledge_bank_id", None)
        if kb_id:
            kb_name = get_knowledge_bank_name(kb_id) 
            if kb_name:
                tool_name = kb_name
    elif retrieval_mode == RetrieverMode.DB.value:
        tool_type = RetrieverMode.DB.value
        table_list = config.get("sql_retrieval_table_list", None)
        if table_list:
            tool_name = ", ".join(table_list)

    return tool_type, tool_name