from typing import Optional

from answers.backend.models.base import VectorStoreType
from answers.solutions.vector_search.generic_vector_search import GenericVectorQuery
from common.backend.utils.dataiku_api import dataiku_api
from dataiku.core import intercom


class VectorSearch:
    def __init__(self, kb_id: Optional[str] = None, vector_store_type: Optional[str] = None) :
        pk: str = dataiku_api.default_project_key
        self.full_kb_id = f"{pk}.{kb_id}"
        self.vector_store_type = vector_store_type
        self._kb_dict: Optional[dict] = None

    def _get_vector_store_type(self):
        if self._kb_dict is None:
            self._kb_dict = intercom.backend_json_call(
                "knowledge-bank/get", data={"knowledgeBankFullId": self.full_kb_id}
            )
            if self._kb_dict is None:
                raise ValueError(f"Knowledge bank with id {self.full_kb_id} not found")
        return self._kb_dict.get("vectorStoreType")

    def get_vector_store(self) -> GenericVectorQuery:
        if not self.vector_store_type:
            self._get_vector_store_type()

        if self.vector_store_type == VectorStoreType.QDRANT_LOCAL.value:
            from answers.solutions.vector_search.qdrant_vector_search import QdrantVectorQuery
            return QdrantVectorQuery()

        elif self.vector_store_type == VectorStoreType.PINECONE.value:
            from answers.solutions.vector_search.pinecone_vector_search import PineConeVectorQuery
            return PineConeVectorQuery()

        elif self.vector_store_type == VectorStoreType.CHROMA.value:
            from answers.solutions.vector_search.chroma_vector_search import ChromaVectorQuery
            return ChromaVectorQuery()

        elif self.vector_store_type == VectorStoreType.AZURE_AI_SEARCH.value:
            from answers.solutions.vector_search.azure_ai_search_search import AzureAiSearchQuery
            return AzureAiSearchQuery()
            # raise NotImplementedError("Azure AI Search based vector search is not supported yet")

        elif self.vector_store_type == VectorStoreType.ELASTICSEARCH.value:
            # from answers.solutions.vector_search.elastic_search_search import ElasticSearchVectorQuery
            # return ElasticSearchVectorQuery()
            raise NotImplementedError("Elasticsearch based vector search is not supported yet")
        
        elif self.vector_store_type == VectorStoreType.VERTEX_AI_GCS_BASED.value:
            # from answers.solutions.vector_search.vertex_ai_gcs_vector_search import VertexAIGCSVectorQuery
            # return VertexAIGCSVectorQuery()
            raise NotImplementedError("Vertex AI GCS based vector search is not supported yet")
        elif self.vector_store_type == VectorStoreType.FAISS.value:
            raise NotImplementedError("FAISS based vector search is not supported")

        else:
            raise ValueError("Unsupported vector store")
