import copy
from typing import List, Union

from common.backend.models.source import RAGImage, Source
from common.backend.utils.picture_utils import b64encode_image_from_full_folder_id
from common.llm_assist.logging import logger


def load_images_data(images: Union[List[RAGImage], List[str]]) -> List[RAGImage]:
    if not isinstance(images, List): # avoid to break if images is None or other.
        return []

    output: List[RAGImage] = []
    for image in images:
        if isinstance(image, dict):
            # RagImage case
            file_path = image.get("file_path") or ""
            full_folder_id = image.get("full_folder_id") or ""
            if isinstance(file_path, str):
                image["file_data"] = b64encode_image_from_full_folder_id(file_path, full_folder_id)
            elif isinstance(file_path, List) and len(file_path) == 1: # before we were recording filepath as a an array so we need to support it.
                image["file_data"] = b64encode_image_from_full_folder_id(file_path[0], full_folder_id)
                logger.debug(f"file_path is an array of one string")
            else:
                logger.warn(f"file_path is not in the expected format : {file_path}")
            output.append(image)
        else:
            # Here we handle legacy
            output.append({"file_data": image})
    return output


def rm_image_data_from_sources_to_store(sources: List[Source]) -> List[Source]:
    # Do not store images data in db
    # Only store info of images to be loaded later

    sources_copy: List[Source] = []
    if sources:
        sources_copy = copy.deepcopy(sources) # type: ignore
        for source in sources_copy:
            if source.get("images"):
                for image in source["images"] or []:
                    image.pop("file_data", None)
    return sources_copy
