import axios from 'axios'
import type { AxiosInstance } from 'axios'
import type {
  QuestionData,
  UISetupResponse,
  ConversationsResponse,
  ConversationResponse,
  UploadResponse,
  LoadFileResponse
} from '@/models'

export class ServerApi {
  public static client: AxiosInstance
  public static errors: any[] = []
  private static _host: string | undefined
  private static initialized = false

  static get host(): string | undefined {
    return this._host
  }

  static set host(value: string | undefined) {
    this._host = value
    this.client = axios.create({ baseURL: value })
    this.client.interceptors.response.use(
      (response) => response,
      (error) => {
        console.error(error)
        this.errors.push(error.response)
        return Promise.reject(error)
      }
    )
  }

  public static init(options?: { host?: string }) {
    if (this.initialized) return

    if (options) {
      const { host } = options
      this.host = host
    }

    this.initialized = true
  }
  public static async getUISetup() {
    const responseData = await this.client.get<UISetupResponse>(`/api/config/get_ui_setup`)
    return responseData?.data
  }

  public static async getUserConversations() {
    const responseData = await this.client.get<ConversationsResponse>(
      '/api/conversation/conversations'
    )
    console.debug(responseData?.data)
    return responseData?.data
  }

  public static async getConversation(id: string) {
    const responseData = await this.client.get<ConversationResponse>(`/api/conversation/${id}`)
    return responseData?.data
  }

  public static async clearHistory(id: string) {
    await this.client.delete(`/api/conversation/${id}/history`)
    return true
  }

  public static async editConversationTitle(id: string, newTitle: string) {
    await this.client.put(`/api/conversation/${id}/title`, {
      new_title: newTitle
    })
    return true
  }

  public static async deleteConversation(id: string) {
    await this.client.delete(`/api/conversation/${id}`)
    return true
  }

  public static async deleteAllConversations() {
    await this.client.delete(`/api/conversation/conversations`)
    return true
  }

  public static async deleteFile(filePath: string[], extractedDocMetadata: string[] | null) {
    await this.client.post(`/api/file/delete`, {
      file_path: filePath,
      metadata_path: extractedDocMetadata
    })
    return true
  }

  public static async loadImage(filePath: string) {
    try{
      const response = await this.client.post<LoadFileResponse>(`/api/file/load`, {
        file_path: filePath,
        file_type: 'image'
      })
      return response?.data
    } catch (error) {
      if (axios.isAxiosError(error)) {
        console.log('Error response in server_api', error.response)
        return error.response?.data
      } else {
        console.error('An unexpected error occurred:', error)
        throw error
      }
    }
  }

  public static async summaryUploadFile(data: FormData) {
    try {
      const response = await this.client.post<UploadResponse>(`/api/file/summary_upload`, data, {
        headers: {
          'Content-Type': 'multipart/form-data'
        }
      })
      return response?.data
    } catch (error) {
      if (axios.isAxiosError(error)) {
        console.log('Error response in server_api', error.response)
        return error.response?.data
      } else {
        console.error('An unexpected error occurred:', error)
        throw error
      }
    }
  }

  public static async uploadFile(data: FormData) {
    try {
      const response = await this.client.post<UploadResponse>(`/api/file/upload`, data, {
        headers: {
          'Content-Type': 'multipart/form-data'
        }
      })
      return response?.data
    } catch (error) {
      if (axios.isAxiosError(error)) {
        console.log('Error response in server_api', error.response)
        return error.response?.data
      } else {
        console.error('An unexpected error occurred:', error)
        throw error
      }
    }
  }

  public static async logGeneralFeedback(data: { message: string; knowledge_bank_id?: string }) {
    await this.client.post(`/api/general_feedback/submit`, data)
    return true
  }

  public static async updateUserProfile(data: any) {
    const response = await this.client.post(`/api/config/user/profile`, data)
    return response.data
  }

  public static async logFeedback(
    conversation_id: string,
    record_id: number | string,
    data: QuestionData
  ) {
    await this.client.post(
      `/api/conversation/${conversation_id}/records/${String(record_id)}/log_feedback`,
      data
    )
    return true
  }

}
