import pandas as pd
import numpy as np
from logging_assist.logging import logger
from time import time
import re

class RelativitiesCalculator:
    """
    A class to handle interactions with a Dataiku model.

    Attributes:
        model_id (str): The ID of the model.
        model (dataiku.Model): The Dataiku model object.
        full_model_id (str): The full model ID of the active model version.
        model_info_handler (PredictionModelInformationHandler): Handler for model information.
    """

    def __init__(self, data_handler, model_retriever, prepared_train_set=None, prepared_test_set=None, base_values=None, modalities=None, variable_types=None):
        """
        Initializes the ModelHandler with a specific model ID.

        Args:
            model_id (str): The ID of the model to handle.
        """
        self.data_handler = data_handler
        self.model_retriever = model_retriever
        try:
            if prepared_train_set is not None:
                self.train_set = prepared_train_set
            else:
                self.train_set = self.prepare_dataset('train')
            if prepared_test_set is not None:
                self.test_set = prepared_test_set
            else:
                self.test_set = self.prepare_dataset('test')
            if base_values is None:
                self.base_values = {}
                self.modalities = {}
                self.variable_types = {}
                self.compute_base_values()
            else:
                self.base_values = base_values
                self.modalities = modalities
                self.variable_types = variable_types
            logger.info("Relativities ModelHandler initialized.")
            logger.info(f"length of train set is {len(self.train_set)}")
        except Exception as e:
            logger.error(f"Error initializing RelativitiesCalculator: {e}")
            self.train_set = None
            self.test_set = None
    
    def _predict_from_df(self, df):
        preprocessed_data = self.model_retriever.predictor.preprocess(df)
        predictions_array = self.model_retriever.predictor._clf.predict(preprocessed_data[0])
        return predictions_array

    def compute_base_values(self):
        logger.info("Computing base values on initiation.")
        params = self.model_retriever.predictor.params
        preprocessing_features = params.preprocessing_params['per_feature']

        for feature, config in preprocessing_features.items():
            self.base_values[feature] = self.extract_base_level(config['customHandlingCode'])
            self.modalities[feature] = self.train_set[feature].unique()
            self.variable_types[feature] = config['type']

        logger.info("Base values computed and modalities extracted.")

    def get_base_values(self):
        logger.info(f"Getting base values")
        return self.base_values

    def extract_base_level(self, custom_code):
        """
        Extracts Base Level from preprocessing custom code.
        Supports string and numeric (int/float) values.
        """
        base_level = None
        # Match either a quoted string or a signed integer/float
        pattern = r'"base_level":\s*(?:"([^"]+)"|([+-]?\d+(?:\.\d+)?))'
        match = re.search(pattern, custom_code)
        base_level = None
        if match:
            if match.group(1) is not None:
                base_level = match.group(1)
            elif match.group(2) is not None:
                # Convert numeric string to float to support decimals
                num_str = match.group(2)
                try:
                    base_level = float(num_str)
                except ValueError:
                    base_level = None
        logger.debug(f"returning base_level {base_level}")
        return base_level

    def initialize_baseline(self):
        logger.info("Starting initialize_baseline method")
        train_row = self.train_set.head(1).copy()
        used_features = self.model_retriever.get_used_features()
        logger.info(f"Used features: {used_features}")
        
        for feature in used_features:
            base_value = self.base_values[feature]
            train_row[feature] = base_value

        if self.model_retriever.exposure_columns is not None:
            train_row[self.model_retriever.exposure_columns] = 1
            logger.debug(f"Exposure column(s) set to 1")
        else:
            logger.info("No exposure columns to set")

        logger.info("Successfully completed initialize_baseline method")
        return train_row

    def calculate_baseline_prediction(self, sample_train_row):
        logger.info("Calculating baseline prediction")
        return self._predict_from_df(sample_train_row)[0]

    def construct_relativities_df(self):
        logger.info("constructing relativites DF")
        rel_df = pd.DataFrame(columns=['feature', 'value', 'relativity'])
        for feature, values in self.relativities.items():
            for value, relativity in values.items():
                rel_df = rel_df.append({'feature': feature, 'value': value, 'relativity': relativity}, ignore_index=True)
        return rel_df
    
    def construct_relativities_interaction_df(self):
        logger.info("constructing relativites DF")
        rel_df = pd.DataFrame(columns=['feature_1', 'feature_2', 'value_1', 'value_2', 'relativity'])
        for feature_1, features in self.relativities_interaction.items():
            for feature_2, values in features.items():
                for value_1, relativities in values.items():
                    for value_2, relativity in relativities.items():
                        rel_df = rel_df.append({'feature_1': feature_1, 
                        'feature_2': feature_2,
                        'value_1': value_1,
                        'value_2': value_2, 
                        'relativity': relativity}, ignore_index=True)
        return rel_df
    
    def get_relativities_df(self):
        """
        Computes and returns the relativities DataFrame for the model.
        (Optimized with batch prediction)
        Returns:
            pd.DataFrame: The relativities DataFrame.
        """
        logger.info("Computing relativities DataFrame.")
        sample_train_row = self.initialize_baseline()
        baseline_prediction = self.calculate_baseline_prediction(sample_train_row)

        self.relativities = {'base': {'base': baseline_prediction}}
        used_features = self.model_retriever.get_used_features()

        dfs_to_predict = []
        features_and_values = [] # To map results back

        for feature in used_features:
            feature_type = self.model_retriever.features[feature]['type']
            base_value = self.base_values[feature]
            self.relativities[feature] = {}

            exposure_col = self.model_retriever.exposure_columns
            exposure_per_modality = self.train_set.groupby(feature)[exposure_col].sum()
            top_modalities = exposure_per_modality.nlargest(99).index.tolist()
            values_to_process = top_modalities
            if base_value not in values_to_process:
                values_to_process.append(base_value)

            for value in values_to_process:
                if value == base_value:
                    self.relativities[feature][value] = 1.0
                    continue
                
                train_row_copy = sample_train_row.copy()
                train_row_copy[feature] = value
                dfs_to_predict.append(train_row_copy)
                features_and_values.append((feature, value))

        if dfs_to_predict:
            logger.info(f"Predicting batch of {len(dfs_to_predict)} rows for relativities...")
            batch_df = pd.concat(dfs_to_predict, ignore_index=True)
            batch_predictions = self._predict_from_df(batch_df)
            
            for i, (feature, value) in enumerate(features_and_values):
                prediction = batch_predictions[i]
                relativity = prediction / baseline_prediction
                self.relativities[feature][value] = relativity

        relativities_df = self.construct_relativities_df()
        logger.info("Relativities DataFrame computed")
        return relativities_df

    def get_relativities_interactions_df(self):
        """
        Computes and returns the relativities DataFrame for the model.
        (Optimized with batch prediction)
        Returns:
            pd.DataFrame: The relativities DataFrame.
        """
        logger.info("Computing relativities DataFrame.")
        sample_train_row = self.initialize_baseline()
        baseline_prediction = self.calculate_baseline_prediction(sample_train_row)

        self.relativities_interaction = {}
        interactions = self.model_retriever.get_interactions()
        
        dfs_to_predict = []
        features_and_values_list = [] # To map results back

        for interaction in interactions:
            interaction_first = interaction[0]
            interaction_second = interaction[1]
            
            base_value_first = self.base_values[interaction_first]
            base_value_second = self.base_values[interaction_second]
            
            # Initialize the nested dictionary structure
            if interaction_first not in self.relativities_interaction:
                self.relativities_interaction[interaction_first] = {}
            if interaction_second not in self.relativities_interaction[interaction_first]:
                self.relativities_interaction[interaction_first][interaction_second] = {}
            if base_value_first not in self.relativities_interaction[interaction_first][interaction_second]:
                 self.relativities_interaction[interaction_first][interaction_second][base_value_first] = {}
            
            # Set base relativity
            self.relativities_interaction[interaction_first][interaction_second][base_value_first][base_value_second] = 1.0
            
            type_first = self.variable_types.get(interaction_first)
            type_second = self.variable_types.get(interaction_second)

            values_to_process_first = self.modalities[interaction_first] if type_first == 'CATEGORICAL' else [base_value_first]
            values_to_process_second = self.modalities[interaction_second] if type_second == 'CATEGORICAL' else [base_value_second]
            
            for value_first in values_to_process_first:
                for value_second in values_to_process_second:
                    if value_first == base_value_first and value_second == base_value_second:
                        continue # Skip base case, already set to 1.0

                    train_row_copy = sample_train_row.copy()
                    train_row_copy[interaction_first] = value_first
                    train_row_copy[interaction_second] = value_second
                    dfs_to_predict.append(train_row_copy)
                    features_and_values_list.append((interaction_first, interaction_second, value_first, value_second))

        # Predict on the entire batch at once
        if dfs_to_predict:
            logger.info(f"Predicting batch of {len(dfs_to_predict)} rows for interactions...")
            batch_df = pd.concat(dfs_to_predict, ignore_index=True)
            batch_predictions = self._predict_from_df(batch_df)
            
            # Map results back
            for i, (f1, f2, v1, v2) in enumerate(features_and_values_list):
                prediction = batch_predictions[i]
                relativity = prediction / baseline_prediction
                if v1 not in self.relativities_interaction[f1][f2]:
                    self.relativities_interaction[f1][f2][v1] = {}
                self.relativities_interaction[f1][f2][v1][v2] = relativity

        relativities_interaction_df = self.construct_relativities_interaction_df()
        logger.info("Relativities DataFrame computed")
        return relativities_interaction_df

    def apply_weights_to_data(self, test_set):
        used_features = self.model_retriever.get_used_features()
        print(f"Using feature list of {used_features}")
        if self.model_retriever.exposure_columns is None:
            test_set['weight'] = 1
        else:
            test_set['weight'] = test_set[self.model_retriever.exposure]
        test_set['weighted_target'] = test_set[self.model_retriever.target_columns] * test_set['weight']
        test_set['weighted_predicted'] = test_set['predicted'] * test_set['weight']

    def prepare_dataset(self, dataset_type='train'):
        """
        Prepares and returns either the training or test dataset.

        Args:
            dataset_type (str): Either 'train' or 'test'. Defaults to 'train'.

        Returns:
            pd.DataFrame: The prepared dataset.
        """
        logger.info(f"Preparing {dataset_type} dataset.")

        if dataset_type == 'train':
            dataset = self.model_retriever.model_info_handler.get_train_df()[0].copy()
        elif dataset_type == 'test':
            dataset = self.model_retriever.model_info_handler.get_test_df()[0].copy()
        else:
            raise ValueError("dataset_type must be either 'train' or 'test'")

        predicted = self._predict_from_df(dataset)
        dataset['predicted'] = predicted
        dataset['weight'] = 1 if self.model_retriever.exposure_columns is None else dataset[self.model_retriever.exposure_columns]

        dataset['weighted_target'] = dataset[self.model_retriever.target_column]
        dataset['weighted_predicted'] = dataset['predicted']
        
        logger.info(f"{dataset_type.capitalize()} dataset prepared: {dataset.shape}")
        return dataset
    
    def compute_base_predictions_variable(self, test_set, used_features, feature, max_modalities=100, grouping_info=None):
        logger.info(f"Starting compute_base_predictions for {feature}")
        base_data = {}
        copy_test_df = test_set.copy()
        copy_test_df[self.model_retriever.exposure_columns] = 1

        feature_type = self.variable_types.get(feature, None)
        exposure_col = self.model_retriever.exposure_columns
        if exposure_col is None:
            copy_test_df['weight'] = 1
            exposure_col = 'weight'

        bin_map = None
        if feature_type == 'CATEGORY':
            exposure_per_modality = copy_test_df.groupby(feature)[exposure_col].sum()
            top_modalities = exposure_per_modality.nlargest(max_modalities - 1).index
            copy_test_df[feature] = copy_test_df[feature].where(copy_test_df[feature].isin(top_modalities), other='Other')
            feature_df = copy_test_df.groupby(feature, as_index=False).first()
            feature_df[exposure_col] = 1
        elif feature_type == 'NUMERIC':
            unique_vals = copy_test_df[feature].nunique(dropna=True)
            if unique_vals > max_modalities:
                if grouping_info is None:
                    copy_test_df['feature_bin'] = pd.qcut(copy_test_df[feature], q=max_modalities, duplicates='drop')
                    def weighted_mean(x):
                        weights = x[exposure_col].fillna(0)
                        weight_sum = weights.sum()
                        if weight_sum <= 0:
                            logger.warning(
                                "Zero-weight bin detected for %s; falling back to unweighted average.",
                                feature
                            )
                            return x[feature].mean()
                        return np.average(x[feature], weights=weights)
                    bin_means = copy_test_df.groupby('feature_bin').apply(weighted_mean)
                    bin_map = dict(zip(bin_means.index, bin_means.values))
                    # Map the bin containing the base_value to the base_value itself
                    base_value = self.base_values.get(feature, None)
                    if base_value is not None:
                        for bin_label in bin_map:
                            left = bin_label.left if hasattr(bin_label, 'left') else None
                            right = bin_label.right if hasattr(bin_label, 'right') else None
                            if left is not None and right is not None and left < base_value <= right:
                                bin_map[bin_label] = float(base_value)
                                break
                else:
                    bin_map = grouping_info
                    copy_test_df['feature_bin'] = pd.qcut(copy_test_df[feature], q=max_modalities, duplicates='drop')
                copy_test_df[feature] = copy_test_df['feature_bin'].map(bin_map).astype(float)
                feature_df = copy_test_df.groupby('feature_bin', as_index=False).first()
                feature_df[feature] = feature_df['feature_bin'].map(bin_map).astype(float)
                feature_df[exposure_col] = 1
                feature_df = feature_df.drop(columns=['feature_bin'])
            else:
                feature_df = copy_test_df.groupby(feature, as_index=False).first()
                feature_df[exposure_col] = 1
        else:
            feature_df = copy_test_df.groupby(feature, as_index=False).first()
            feature_df[exposure_col] = 1

        for other_feature in used_features:
            if other_feature != feature:
                feature_df[other_feature] = self.base_values[other_feature]

        predictions = self._predict_from_df(feature_df)
        base_data[feature] = pd.DataFrame({
            f'base_{feature}': predictions,
            feature: feature_df[feature]
        })

        logger.info("Finished compute_base_predictions")
        return base_data, bin_map
    
    def get_formated_predicted_base(self):
        logger.info("Getting formatted and predicted base")
        self.get_predicted_and_base()
        df = self.predicted_base_df.copy()
        df.columns = ['definingVariable', 
                     'Category', 
                     'observedAverage', 
                     'fittedAverage', 'Value', 'baseLevelPrediction', 'dataset']
        logger.info("Successfully got formatted and predicted base")
        return df
    
    def get_formated_predicted_base_variable(self, variable):
        logger.info("Getting formatted and predicted base")
        self.get_predicted_and_base_variable(variable)
        df = self.predicted_base_df.copy()
        df.columns = ['definingVariable', 
                     'Category', 
                     'observedAverage', 
                     'fittedAverage', 'Value', 'baseLevelPrediction', 'dataset']
        logger.info("Successfully got formatted and predicted base")
        return df
    
    def merge_predictions(self, test_set, base_data):
        logger.info("Merging Base predictions")
        for feature in base_data.keys():
            test_set = test_set.set_index(feature).join(base_data[feature].set_index(feature), how='left')
        logger.info("Successfully Merged Base predictions")
        return test_set
    
    def process_dataset_variable(self, dataset, dataset_name, variable, max_modalities=100):
        logger.info(f"Processing dataset {dataset_name}")
        used_features = self.model_retriever.get_used_features()
        # Compute base_data and get grouping info
        base_data, bin_map = self.compute_base_predictions_variable(dataset, used_features, variable, max_modalities=max_modalities)
        feature_type = self.variable_types.get(variable, None)
        exposure_col = self.model_retriever.exposure_columns or 'weight'
        grouped_dataset = dataset.copy()
        if feature_type == 'CATEGORY':
            exposure_per_modality = grouped_dataset.groupby(variable)[exposure_col].sum()
            top_modalities = exposure_per_modality.nlargest(max_modalities - 1).index
            grouped_dataset[variable] = grouped_dataset[variable].where(grouped_dataset[variable].isin(top_modalities), other='Other')
        elif feature_type == 'NUMERIC':
            unique_vals = grouped_dataset[variable].nunique(dropna=True)
            if unique_vals > max_modalities and bin_map is not None:
                grouped_dataset['feature_bin'] = pd.qcut(grouped_dataset[variable], q=max_modalities, duplicates='drop')
                grouped_dataset[variable] = grouped_dataset['feature_bin'].map(bin_map).astype(float)
                grouped_dataset = grouped_dataset.drop(columns=['feature_bin'])
        dataset = self.merge_predictions(grouped_dataset, base_data)
        predicted_base = self.data_handler.calculate_weighted_aggregations(dataset, [variable], ([variable] if variable in used_features else []))
        predicted_base_df = self.data_handler.construct_final_dataframe(predicted_base)
        predicted_base_df['dataset'] = dataset_name
        logger.info(f"Processed dataset {dataset_name}")
        return predicted_base_df
    
    def get_predicted_and_base(self, nb_bins_numerical=100000):
        logger.info("Getting Predicted and base")
        
        test_predictions = self.process_dataset(self.test_set, 'test')
        train_predictions = self.process_dataset(self.train_set, 'train')
        
        self.predicted_base_df = train_predictions.append(test_predictions)
        categorical_variables = [variable for variable in self.variable_types.keys() if self.variable_types[variable] == 'CATEGORY']
        self.predicted_base_df['category'] = [str(category) if variable in categorical_variables else category for category, variable in zip(self.predicted_base_df['category'], self.predicted_base_df['feature'])]
        logger.info("Successfully got Predicted and base")
        return self.predicted_base_df.copy()
    
    def get_predicted_and_base_variable(self, variable, nb_bins_numerical=100000):
        logger.info(f"Getting Predicted and base for variable {variable}")
        
        test_predictions = self.process_dataset_variable(self.test_set, 'test', variable)
        train_predictions = self.process_dataset_variable(self.train_set, 'train', variable)
        
        self.predicted_base_df = train_predictions.append(test_predictions)
        categorical_variables = [variable for variable in self.variable_types.keys() if self.variable_types[variable] == 'CATEGORY']
        self.predicted_base_df['category'] = [str(category) if variable in categorical_variables else category for category, variable in zip(self.predicted_base_df['category'], self.predicted_base_df['feature'])]
        logger.info("Successfully got Predicted and base")
        return self.predicted_base_df.copy()


