# -*- coding: utf-8 -*-
import dataiku
from dataiku.customrecipe import (
    get_recipe_config,
    get_input_names_for_role,
    get_output_names_for_role,
)
from osm_dataset_enrichment.config.dku_config import DkuConfig


class DkuConfigLoading:
    """Generic class to instanciate a dku_config object to store and check input parameters"""

    def __init__(self):
        self.config = get_recipe_config()
        self.dku_config = DkuConfig()


class DkuConfigLoadingOSM(DkuConfigLoading):
    """Configuration for Ontology Tagging Plugin"""

    def __init__(self):
        """Instanciate class with DkuConfigLoading and add input datasets to dku_config"""
        super().__init__()
        polygon_dataset = get_input_names_for_role("polygon_dataset")[0]
        self.dku_config.add_param(
            name="polygon_dataset",
            value=dataiku.Dataset(polygon_dataset),
            required=True,
        )
        self.polygon_dataset_columns = [p["name"] for p in self.dku_config.polygon_dataset.read_schema()]

    def _content_error_message(self, error, column):
        """Get corresponding error message if any"""
        if error == "missing":
            return f"Missing input column."

        if error == "invalid":
            return f"Invalid input column : {column}.\n"

        if error == "missing_values":
            return f"The recipe requires at least one value in the '{column}' field.\n"

    def _add_polygon_column(self):
        """Load text column from Geometry Dataset"""
        polygon_column = self.config.get("polygon_column")
        input_dataset_columns = self.polygon_dataset_columns
        self.dku_config.add_param(
            name="polygon_column",
            value=polygon_column,
            required=True,
            checks=self._get_column_checks(polygon_column, input_dataset_columns),
        )

    def _add_type_of_pois(self):
        """Load the given tags to filter pois with"""
        filters = list(set(self.config.get("type_of_pois")))
        self.dku_config.add_param(
            name="type_of_pois",
            value=filters,
            required=True,
            checks=[
                {
                    "type": "custom",
                    "cond": len(filters) > 0,
                    "err_msg": self._content_error_message("missing_values", "Type of POIs"),
                }
            ],
        )

    def _add_enrichment_keys(self):
        """Load enrichment_keys parameters"""
        pois_enrichment = self.config.get("pois_enrichment")
        if pois_enrichment:
            enrichment_keys = list(set(self.config.get("enrichment_keys")))
        else:
            enrichment_keys = []
        self.dku_config.add_param(name="enrichment_keys", value=enrichment_keys, required=False)

    def _add_request_by_batch(self):
        request_by_batch = self.config.get("request_by_batch")
        self.dku_config.add_param(name="request_by_batch", value=request_by_batch, required=True)

    def _get_column_checks(self, column, input_columns):
        """Check for polygon_column parameter"""
        return [
            {
                "type": "exists",
                "err_msg": self._content_error_message("missing", None),
            },
            {
                "type": "in",
                "op": input_columns,
                "err_msg": self._content_error_message("invalid", column),
            },
        ]

    def _add_output_dataset(self):
        """Load the output dataset"""
        output_dataset_name = get_output_names_for_role("pois_dataset")[0]
        self.dku_config.add_param(
            name="pois_dataset",
            value=dataiku.Dataset(output_dataset_name),
            required=True,
        )

    def load_settings(self):
        """Public function to load all input parameters and return them through dku_config object"""
        self._add_polygon_column()
        self._add_type_of_pois()
        self._add_enrichment_keys()
        # self._add_request_mode()
        self._add_request_by_batch()
        self._add_output_dataset()
        return self.dku_config
