import dataiku


def get_current_project_and_variables():
    project_key = dataiku.get_custom_variables()["projectKey"]
    client = dataiku.api_client()
    project = client.get_project(project_key)
    variables = project.get_variables()
    return project, variables


def switch_recipe_engine(project, recipe_name, new_engine):
    recipe = project.get_recipe(recipe_name)
    recipe_settings = recipe.get_settings()
    recipe_type = recipe_settings.type
    if recipe_type in ["prepare", "shaker"]:
        recipe_settings.get_recipe_params()["engineType"] = new_engine

    elif recipe_type == "split":
        recipe_settings.obj_payload["engineType"] = new_engine

    else:
        recipe_settings.get_json_payload()["engineType"] = new_engine
    recipe_settings.save()
    pass


def update_recipe_ouput_schema(project, recipe_name, run=False):
    recipe = project.get_recipe(recipe_name)
    required_updates = recipe.compute_schema_updates()
    if required_updates.any_action_required():
        required_updates.apply()
    if run:
        recipe.run()


def change_dataset_connection_to_sql(
    project, dataset_name, new_connection_name, new_connection_type, new_connection_schema
):
    dataset = project.get_dataset(dataset_name)
    dataset_settings = dataset.get_settings()
    dataset_settings.settings["params"]["connection"] = new_connection_name
    dataset_settings.settings["params"]["mode"] = "table"
    dataset_settings.settings["params"]["schema"] = new_connection_schema
    dataset_settings.settings["type"] = new_connection_type
    dataset_settings.save()
    pass


# def get_recipe_available_engines(project, recipe_name):
#     available_engines = []
#     while len(available_engines) == 0:
#         recipe = project.get_recipe(recipe_name)
#         recipe_status = recipe.get_status()
#         recipe_engine_details = recipe_status.get_engines_details()
#         available_engines = [entity["type"] for entity in recipe_engine_details if (entity["isSelectable"] == True)]
#         if len(available_engines) == 0:
#             switch_recipe_engine(project, recipe_name, "DSS")
#     return available_engines


def get_recipe_output_datasets(project, recipe_name):
    recipe = project.get_recipe(recipe_name)
    recipe_settings = recipe.get_settings()
    recipe_output_items = recipe_settings.get_recipe_outputs()["main"]["items"]
    return [item["ref"] for item in recipe_output_items]


def get_recipe_input_datasets(project, recipe_name):
    recipe = project.get_recipe(recipe_name)
    recipe_settings = recipe.get_settings()
    recipe_input_items = recipe_settings.get_recipe_inputs()["main"]["items"]
    return [item["ref"] for item in recipe_input_items]


def get_all_projects_datasets(project):
    project_datasets = project.list_datasets()
    return [metadata["name"] for metadata in project_datasets]


def copy_sql_dataset_connection_settings(project, dataset_to_update_name, reference_dataset_name, default_table_name):
    dataset_to_update = project.get_dataset(dataset_to_update_name)
    dataset_to_update_settings = dataset_to_update.get_settings()

    reference_dataset = project.get_dataset(reference_dataset_name)
    reference_dataset_settings = reference_dataset.get_settings()

    dataset_to_update_settings.settings["params"]["table"] = default_table_name
    dataset_to_update_settings.settings["params"]["mode"] = reference_dataset_settings.settings["params"]["mode"]
    dataset_to_update_settings.settings["params"]["schema"] = reference_dataset_settings.settings["params"]["schema"]
    dataset_to_update_settings.save()


def change_dataset_managed_state(project, dataset_name, bool_should_be_managed_state):
    dataset = project.get_dataset(dataset_name)
    dataset_settings = dataset.get_settings()
    dataset_settings.settings["managed"] = bool_should_be_managed_state
    dataset_settings.save()


def change_sql_dataset_table_naming_strategy(project, dataset_name):
    project_key = project.project_key
    dataset = project.get_dataset(dataset_name)
    dataset_settings = dataset.get_settings()
    dataset_settings.settings["params"]["table"] = "{}_{}".format(project_key, dataset_name)
    dataset_settings.save()


def existing_recipe(project, input_name, output_name):
    if output_name not in get_all_projects_datasets(project):
        return None
    output_dataset = project.get_dataset(output_name)
    recipe_dict = next((recipe for recipe in output_dataset.get_usages() if recipe["type"] == "RECIPE_OUTPUT"), None)
    if recipe_dict is not None:
        recipe = project.get_recipe(recipe_dict["objectId"])
        if recipe.get_settings().has_input(input_name):
            return recipe
        else:
            raise ValueError(
                "Cannot create a recipe if the output dataset ('{}') is already used as an output".format(output_name)
            )
    return None


def create_new_managed_dataset(project, dataset_name, connection_id):
    if dataset_name not in [dataset.name for dataset in project.list_datasets(as_type="objects")]:
        builder = project.new_managed_dataset_creation_helper(dataset_name)
        builder.with_store_into(connection_id)
        builder.create()


def create_prepare_recipe(project, input_name, output_name, connection_id):
    recipe = existing_recipe(project, input_name, output_name)
    if recipe is not None:
        return recipe.name
    prepare_recipe = project.new_recipe("prepare")
    prepare_recipe.with_input(input_name)
    prepare_recipe.with_new_output(output_name, connection_id)
    recipe = prepare_recipe.create()
    return recipe.name


def edit_prepare_recipe(project, recipe_name, renaming_map):
    recipe = project.get_recipe(recipe_name)
    recipe_settings = recipe.get_settings()
    json_payload = recipe_settings.get_json_payload()
    json_payload["steps"] = []

    recipe_settings.add_processor_step(
        type="ColumnsSelector", params={"appliesTo": "COLUMNS", "columns": list(renaming_map.keys()), "keep": True}
    )

    recipe_settings.add_processor_step(
        type="ColumnRenamer", params={"renamings": [{"from": key, "to": value} for key, value in renaming_map.items()]}
    )

    recipe_settings.save()


def create_topn_recipe(project, input_name, output_name, connection_id):
    recipe = existing_recipe(project, input_name, output_name)
    if recipe is not None:
        return recipe.name

    create_new_managed_dataset(project, output_name, connection_id)

    topn_recipe = project.new_recipe("topn")
    topn_recipe.with_input(input_name)
    topn_recipe.with_output(output_name)
    recipe = topn_recipe.create()
    return recipe.name


def edit_topn_recipe(project, recipe_name, top_n, ranking_column):
    recipe = project.get_recipe(recipe_name)
    recipe_settings = recipe.get_settings()
    json_payload = recipe_settings.get_json_payload()

    json_payload["firstRows"] = top_n
    json_payload["keys"] = ["user_id"]
    json_payload["orders"] = [{"column": ranking_column, "desc": True}]
    json_payload["retrievedColumns"] = ["item_id", "user_id", ranking_column]

    recipe_settings.set_json_payload(json_payload)
    recipe_settings.save()


def create_join_recipe(project, input_name_0, input_name_1, output_name):
    recipe = existing_recipe(project, input_name_0, output_name)
    if recipe is not None:
        return recipe.name

    join_recipe = project.new_recipe("join")
    join_recipe.with_input(input_name_0)
    join_recipe.with_input(input_name_1)
    join_recipe.with_existing_output(output_name)
    recipe = join_recipe.create()
    return recipe.name


def edit_join_recipe(project, recipe_name, input_column_0, input_column_1, distinct_left=True):
    recipe = project.get_recipe(recipe_name)
    recipe_settings = recipe.get_settings()
    json_payload = recipe_settings.get_json_payload()
    json_payload["joins"] = []
    if distinct_left:
        json_payload["virtualInputs"][0]["preFilter"] = {'distinct': True, 'enabled': False}

    join = recipe_settings.add_join(join_type="LEFT", input1=0, input2=1)
    recipe_settings.add_condition_to_join(join, type="EQ", column1=input_column_0, column2=input_column_1)
    recipe_settings.save()


def set_subset_schema(project, dataset_name, schema_map):
    dataset = project.get_dataset(dataset_name)
    old_schema = dataset.get_schema()
    new_schema = [
        column_schema
        if column_schema["name"] not in schema_map
        else {"name": column_schema["name"], "type": schema_map[column_schema["name"]]}
        for column_schema in old_schema["columns"]
    ]
    dataset.set_schema({"columns": new_schema})


def move_datasets_to_zone(project, dataset_names, zone_id):
    for dataset_name in dataset_names:
        project.get_dataset(dataset_name).move_to_zone(zone_id)
