import dataiku
from dataiku.sql import Dialects
from packaging.version import parse as parse_version
from typing import Any, Dict, List

ALLOWED_CONN_TYPES = [
    Dialects.POSTGRES,
    Dialects.SNOWFLAKE,
    Dialects.REDSHIFT,
    Dialects.SQLSERVER,
    Dialects.BIGQUERY,
    Dialects.DATABRICKS,
    Dialects.ORACLE,
    Dialects.MYSQL,
]


def do(payload: Dict, config: Dict, plugin_config, inputs) -> Dict:
    """Entry point for the plugin's settings backend, routing requests for dynamic UI choices."""
    if payload.get("parameterName") == "datasets":
        return list_conn_datasets(config)
    elif payload.get("parameterName") == "enduser_sql_execution":
        return list_enduser_sql_execution(config)
    return {"choices": []}


def list_enduser_sql_execution(
    config: Dict[str, Any],
) -> Dict[str, List[Dict[str, str]]]:
    """List possible configuration of user executing SQL queries"""
    choices = [{"value": "tool_user", "label": "User calling the tool"}]
    client = dataiku.api_client()
    dss_version_str = client.get_instance_info().raw.get("dssVersion", "0.0")

    if parse_version(dss_version_str) < parse_version("14.3.2-rc1"):
        return {"choices": choices}
    choices.append({"value": "enduser", "label": "End-user caller (through tool context)"})
    choices.append({"value": "enduser_available", "label": "End-user caller if available, else user calling the tool"})
    return {"choices": choices}


def list_conn_datasets(config: Dict[str, Any]) -> Dict[str, List[Dict[str, str]]]:
    """List datasets of a selected connection."""
    connection = config.get("connection")
    if not connection:
        return {"choices": []}  # Return empty if no connection is selected yet.

    datasets = get_connection_datasets(connection)
    return {"choices": [{"value": name, "label": name} for name in datasets]}


def get_project_datasets() -> List[Dict[str, Any]]:
    """Fetches metadata for all datasets in the current Dataiku project."""
    client = dataiku.api_client()
    project = client.get_project(dataiku.default_project_key())
    return project.list_datasets()


def get_connection_datasets(connection: str) -> List[str]:
    """Returns a list of dataset names that belong to the specified connection."""
    return [
        dataset["name"]
        for dataset in get_project_datasets()
        if (
            dataset.get("params", {}).get("connection") == connection
            and dataset.get("type", None) in ALLOWED_CONN_TYPES
        )
    ]
