import curses
from typing import Any, Callable, Dict, List, Set

from dataikuapi.dss.project import DSSAgentToolListItem


def display_selector(
    stdscr: curses.window,
    items: List[Any],
    title: str,
    instructions: str,
    get_display_text: Callable[[Any, str], str],
    scrollable: bool = True,
) -> List[Any]:
    curses.curs_set(0)
    curses.start_color()
    curses.init_pair(1, curses.COLOR_WHITE, curses.COLOR_BLUE)
    curses.init_pair(2, curses.COLOR_WHITE, curses.COLOR_BLACK)
    curses.init_pair(3, curses.COLOR_GREEN, curses.COLOR_BLACK)

    selected_index: int = 0
    checked_items: Set[int] = set()
    scroll_offset: int = 0

    while True:
        stdscr.clear()
        height, width = stdscr.getmaxyx()

        stdscr.addstr(0, (width - len(title)) // 2, title)
        stdscr.addstr(1, (width - len(instructions)) // 2, instructions)

        start_y: int = 3
        end_y: int = height - 2
        display_height: int = end_y - start_y

        visible_items: List[Any] = items[scroll_offset : scroll_offset + display_height]
        for i, item in enumerate(visible_items):
            y_pos: int = start_y + i
            actual_index: int = scroll_offset + i

            if y_pos >= end_y:
                break

            checkbox: str = "[✓]" if actual_index in checked_items else "[ ]"
            display_text: str = get_display_text(item, checkbox)

            if actual_index == selected_index:
                stdscr.attron(curses.color_pair(1))
                stdscr.addstr(y_pos, 2, display_text)
                stdscr.attroff(curses.color_pair(1))
            else:
                if actual_index in checked_items:
                    stdscr.attron(curses.color_pair(3))
                    stdscr.addstr(y_pos, 2, display_text)
                    stdscr.attroff(curses.color_pair(3))
                else:
                    stdscr.attron(curses.color_pair(2))
                    stdscr.addstr(y_pos, 2, display_text)
                    stdscr.attroff(curses.color_pair(2))

        if scrollable and len(items) > display_height:
            if scroll_offset > 0:
                stdscr.addstr(start_y - 1, 2, "↑ More above")
            if scroll_offset + display_height < len(items):
                stdscr.addstr(end_y + 1, 2, "↓ More below")

        bottom_text: str = f"Selected: {len(checked_items)} items"
        stdscr.addstr(height - 1, 0, bottom_text)

        stdscr.refresh()

        key: int = stdscr.getch()

        if key == curses.KEY_UP and selected_index > 0:
            selected_index -= 1
            if selected_index < scroll_offset:
                scroll_offset = selected_index
        elif key == curses.KEY_DOWN and selected_index < len(items) - 1:
            selected_index += 1
            if selected_index >= scroll_offset + display_height:
                scroll_offset = selected_index - display_height + 1
        elif key == ord(" "):
            if selected_index in checked_items:
                checked_items.remove(selected_index)
            else:
                checked_items.add(selected_index)
        elif key == ord("\n") or key == ord("\r"):
            break
        elif key == ord("q"):
            return []

    return [items[i] for i in checked_items]


def display_tool_selector(
    stdscr: curses.window, tools: List[DSSAgentToolListItem]
) -> List[DSSAgentToolListItem]:
    title: str = (
        "Select Tools (Use arrow keys to navigate, space to select, enter to confirm)"
    )
    instructions: str = (
        "↑↓: Navigate | Space: Select/Deselect | Enter: Confirm | q: Quit"
    )

    def get_display_text(tool: DSSAgentToolListItem, checkbox: str) -> str:
        return f"{checkbox} {tool['name']} ({tool['id']})"

    return display_selector(stdscr, tools, title, instructions, get_display_text)


def display_query_group_selector(
    stdscr: curses.window, query_groups: List[Dict[str, Any]]
) -> List[Dict[str, Any]]:
    title: str = (
        "Select Query Groups (Use arrow keys to navigate, space to select, enter to confirm)"
    )
    instructions: str = (
        "↑↓: Navigate | Space: Select/Deselect | Enter: Confirm | q: Quit"
    )

    def get_display_text(group: Dict[str, Any], checkbox: str) -> str:
        db_id: str = group["db_id"]
        count: int = group["count"]
        return f"{checkbox} {db_id} ({count} queries)"

    return display_selector(stdscr, query_groups, title, instructions, get_display_text)
