# === SQLite3 ===
__import__('pysqlite3')
import sys
sys.modules['sqlite3'] = sys.modules.pop('pysqlite3')

# === Third-party Libraries ===
import dataiku
from dataiku.llm.python import BaseLLM

# === CrewAI Framework ===
from crewai import Agent, Task, Crew, LLM
from crewai.tools import tool


@tool("check_order_status")
def check_order_status(order_id: str) -> str: 
    """Checks the status of a specific order ID."""
    # Mock database return
    return f"Order {order_id} is currently Out for Delivery. Expected arrival: Today by 5 PM."

@tool("upgrade_subscription")
def upgrade_subscription(tier: str) -> str:
    """Upgrades the user's plan to the specified tier."""
    return f"Success! The plan has been upgraded to the {tier} Tier."


class MyLLM(BaseLLM):
    def __init__(self):
        # Instantiate the LLM
        LLM_ID = "openai/" + dataiku.get_custom_variables()["LLM_ID"]
        
        # Other method
        BASE_URL = "https://design.ds-platform.ondku.net/public/api/projects/AGENTSFRAMEWORKS/llms/openai/v1/"

        auth_info = dataiku.api_client().get_auth_info(with_secrets=True)
        for secret in auth_info["secrets"]:
            if secret["key"] == "api_key":
                API_KEY = secret["value"]
                break

        llm = LLM(
            model=LLM_ID,
            base_url=BASE_URL,
            api_key=API_KEY
        )
        
        # Define the Agents
        shipping_agent = Agent(
            role="Shipping",
            goal="You are the Shipping. Use tools to check order status.",
            backstory="Once you have answered the shipping question, REVIEW the user's original request.",
            tools=[check_order_status],
            allow_delegation=True,
            verbose=True,
            llm=llm
        )
        
        billing_agent = Agent(
            role="Billing",
            goal="You are the Billing. Help the user upgrade their plan.",
            backstory="Our Gold Tier is currently 20% off. Propose it.",
            tools=[upgrade_subscription],
            allow_delegation=True,
            verbose=True,
            llm=llm
        )
        
        # Define the task
        general_task = Task(
            description=(
                "Answer the request of the answer: {inquiry}"
                "You can answer request about the shipping or the billing subscription."
                "Rely on the shipping or billing agent for specific tasks"
                "Be concise and informative."
                "Don't hallucinate information."
                "Collaborate with your teammates to ensure the completion of the request."
            ),
            agent=shipping_agent,
            expected_output=(
                "Inquiry satisfied, request answered"
            )
        )
        
        # Define the crew
        self.crew = Crew(
            agents=[shipping_agent, billing_agent],
            tasks=[general_task],
            memory=False,
            cache=False,
            verbose=True,
        )

    def process(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
        inputs = {
            "inquiry": prompt
        }
        # Invoke the Agent
        resp_text = self.crew.kickoff(inputs=inputs)
        return {"text": resp_text.raw}
    