# === Dataiku Libraries ===
import dataiku
from dataiku.llm.python import BaseLLM

# === Standard / Python typing ===
from typing import Dict

# === LlamaIndex / Agent Integrations ===
from llama_index.llms.openai_like import OpenAILike
from llama_index.core.tools import FunctionTool
from llama_index.core.agent.workflow import AgentWorkflow, FunctionAgent

# === Project Tools ===
from tools import check_order_status, upgrade_subscription


check_order_status_tool = FunctionTool.from_defaults(check_order_status)
upgrade_subscription_tool = FunctionTool.from_defaults(upgrade_subscription)

# --- Agent System Class ---
class MyLLM(BaseLLM):
    def __init__(self):
        # Retrieve LLM configuration from Dataiku project variables
        project = dataiku.Project()
        LLM_ID = project.get_variables()["standard"]["LLM_ID"]
        
        # Initialize BASE_URL and API_KEY variables
        BASE_URL = "https://design.ds-platform.ondku.net/public/api/projects/MULTIAGENTSINDSS/llms/openai/v1/"

        auth_info = dataiku.api_client().get_auth_info(with_secrets=True)
        for secret in auth_info["secrets"]:
            if secret["key"] == "api_key":
                API_KEY = secret["value"]
                break
        
        llm = OpenAILike(
            model=LLM_ID,
            api_base=BASE_URL,
            api_key=API_KEY,
            context_window=400000,
            is_chat_model=True,
            is_function_calling_model=True,
        )
        
        # Shipping Agent
        shipping_agent = FunctionAgent(
            name="shipping_agent",
            description="Answer shipping related requests.",
            system_prompt="""You are the Shipping Agent. Use tools to check order status.
            Once you have answered the shipping question, REVIEW the user's original request.
            If the user also asked about subscriptions/billing, transfer to billing_agent.
            Answer general message otherwise.
            """,
            tools=[check_order_status_tool],
            llm=llm,
            can_handoff_to=["billing_agent"],
        )

        # Billing Agent
        billing_agent = FunctionAgent(
            name="billing_agent",
            description="Answer billing or subscription related requests.",
            system_prompt="""You are the Billing Agent in charge if subscription and billing requests. 
            Help the user upgrade their plan.
            Our Gold Tier is currently 20% off. Propose it.
            If the user also asked about shipping related questions, transfer to shipping_agent.
            Answer general message otherwise.""",
            tools=[upgrade_subscription_tool],
            llm=llm,
            can_handoff_to=["shipping_agent"],
        )
                
        # Agent Manager
        self.agent = AgentWorkflow(
            agents=[shipping_agent, billing_agent],
            root_agent="shipping_agent" 
        )

    
    async def aprocess(self, query: Dict, settings: Dict, trace: Dict) -> Dict:
        """
        Process a user query asynchronously via the agent workflow.

        Args:
            query (Dict): The input message containing user query.
            settings (Dict): Additional settings (not used here).
            trace (Dict): Trace object for logging/debugging (not used here).

        Returns:
            Dict: A dictionary containing the final response text.
        """
        prompt = query["messages"][-1]["content"]
        
        # Invoke the Agent
        response = await self.agent.run(user_msg=prompt)
        return {"text": response.response.blocks[0].text}

