# === 1. System & Compatibility ===
import sys
from typing import List, Dict

# Necessary hack for CrewAI / SQLite in certain environments
__import__('pysqlite3')
sys.modules['sqlite3'] = sys.modules.pop('pysqlite3')

# === 2. Dataiku ===
import dataiku
from dataiku.llm.python import BaseLLM

# === 3. CrewAI ===
from crewai import Agent, Task, Crew, Process, LLM
from crewai.tools import tool

# === 4. Custom Project Tools ===
from tools import web_search


@tool("web_search")
def web_search(search_query: str) -> List[Dict]:
    """
    Performs a Google search.
    """
    web_search_tool = dataiku.api_client().get_default_project().get_agent_tool("K6yKQTA")
    output = web_search_tool.run({
        "q": search_query
    })
    result = output["output"]
    return result


class MyLLM(BaseLLM):
    def __init__(self):
        # Instantiate the LLM
        LLM_ID = "openai/" + dataiku.get_custom_variables()["LLM_ID"]
        
        # Other method
        BASE_URL = "https://design.ds-platform.ondku.net/public/api/projects/AGENTSFRAMEWORKS/llms/openai/v1/"

        auth_info = dataiku.api_client().get_auth_info(with_secrets=True)
        for secret in auth_info["secrets"]:
            if secret["key"] == "api_key":
                API_KEY = secret["value"]
                break
        
        # Instantiate the LLM
        llm = LLM(
            model=LLM_ID,
            temperature=0,
            base_url=BASE_URL,
            api_key=API_KEY
        )
        
        # Define the sub-agents
        product_scout_agent = Agent(
            role="Product Researcher",
            goal="Find facts about features, pricing, and specs for the competitor during the specified timeframe.",
            backstory="Expert at official product pages and technical changelogs.",
            tools=[web_search],
            llm=llm
        )

        marketing_intel_agent = Agent(
            role="Marketing Researcher",
            goal="Find press releases and news for the competitor during the specified timeframe.",
            backstory="Specialist in tracking brand messaging and public relations.",
            tools=[web_search],
            llm=llm
        )

        sentiment_analyst_agent = Agent(
            role="Sentiment Analyst",
            goal="Analyze forum discussions and reviews for the competitor during the specified timeframe.",
            backstory="Expert in social listening on Reddit and Trustpilot.",
            tools=[web_search],
            llm=llm
        )

        # 2. Define the Explicit Supervisor Agent
        # This agent handles Phase 1 (Planning) and Phase 4 (Synthesis)
        supervisor_agent = Agent(
            role="Competitive Intelligence Supervisor",
            goal="Validate user requests and orchestrate reports.",
            backstory="""You are a professional gatekeeper. 
            - If the user says 'Hello' or provides an incomplete request, DO NOT trigger your workers. 
            - Instead, reply politely and ask for the specific Competitor Name and Timeframe.
            - Only when you have BOTH pieces of information should you begin Phase 1 (Planning).""",
            allow_delegation=True,
            llm=llm
        )

        # 3. Define the Manager Task (The Report Template)
        reporting_task = Task(
            description="""Compile the findings from the three agents into the final report.
            You must use the following template:

            Q3 Competitive Intelligence Report: name of the competitor
            Reporting Period: timeframe

            1. Executive Summary: [3-sentence synthesis]
            2. Product Intelligence: [Map Product_Scout data]
            3. Marketing Strategy: [Map Marketing_Intel data]
            4. Market Sentiment: [Map Sentiment_Analyst data]
            5. Strategic Synthesis: [Threat Level, Q4 Prediction, Counter-Strategy]
            """,
            expected_output="A full Markdown formatted competitive analysis report.",
            agent=supervisor_agent 
        )

        # 4. Initialize the Crew with a Manager LLM
        self.crew = Crew(
            agents=[supervisor_agent, product_scout_agent, marketing_intel_agent, sentiment_analyst_agent],
            tasks=[reporting_task],
            process=Process.hierarchical, 
            manager_llm=llm,  
            verbose=True
        )

    def process(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
        inputs = {
            "inquiry": prompt
        }
        # Invoke the agent
        resp_text = self.crew.kickoff(inputs=inputs)

        return {"text": resp_text.raw}
    