# === Dataiku Imports ===
import dataiku

# === Standard / Python librairies ===
from typing import Literal
import os

# == Dataiku-specific LLM integrations ==
from dataiku.llm.python import BaseLLM

# == LangGraph and LangChain integrations ==
from langchain_core.messages import SystemMessage
from langgraph.graph import StateGraph, MessagesState, START, END
from langgraph.prebuilt import create_react_agent
from langchain_core.tools import tool


@tool
def check_server_status(service_name: str):
    """Checks the health status of a specific technical service (e.g., 'email', 'login', 'database')."""
    # Mock response
    if "email" in service_name.lower():
        return "Service Status: DOWN. Estimated fix time: 2 hours."
    return "Service Status: OPERATIONAL."

@tool
def look_up_invoice(invoice_id: str):
    """Retrieves details for a specific invoice ID."""
    # Mock response
    return f"Invoice {invoice_id}: Status 'Paid', Amount $50.00, Date: 2023-10-01."

@tool
def get_corporate_style_guide():
    """Retrieves the official formatting rules for customer communication."""
    return "Rules: 1. Be polite. 2. Use the phrase 'Happy to help'. 3. Sign off with 'The Team'."

class MyLLM(BaseLLM):
    def __init__(self):
        # Set the LLM
        LLM_ID = dataiku.get_custom_variables().get("LLM_ID", "your_llm_id")
        client = dataiku.api_client()
        project = client.get_default_project()
        llm = project.get_llm(LLM_ID).as_langchain_chat_model()
        
        # Tech Agent: Can check servers
        tech_agent = create_react_agent(
            model=llm, 
            tools=[check_server_status],
            prompt="You are a Tech Support Agent. Use your tools to diagnose issues."
        )

        # Billing Agent: Can look up money
        billing_agent = create_react_agent(
            model=llm, 
            tools=[look_up_invoice],
            prompt="You are a Billing Agent. Use your tools to find invoice details."
        )

        # Final Agent: Checks style and formats
        final_agent = create_react_agent(
            model=llm, 
            tools=[get_corporate_style_guide],
            prompt="You are the Final Quality Assurance Agent. Aggregate the previous information and format it according to the corporate style guide."
        )

        # Define the Router Logic
        def router_node(state: MessagesState):
            """
            The Router is simple: it looks at the last message and decides the path.
            We don't need a full ReAct agent here, just a classification decision.
            """
            messages = state["messages"]
            last_user_msg = messages[-1].content

            # Simple classification prompt
            prompt = f"""
            Classify this query into 'technical' or 'billing'. 
            Return ONLY the label.
            Query: {last_user_msg}
            """
            response = llm.invoke(prompt)
            category = response.content.strip().lower()

            # We return a comprehensive message to guide the next agent
            # Note: In MessagesState, returning a list adds to the existing history
            if "billing" in category:
                return {"messages": [SystemMessage(content="ROUTER DECISION: Routing to Billing Agent.")]}
            else:
                return {"messages": [SystemMessage(content="ROUTER DECISION: Routing to Technical Agent.")]}

        def get_route(state: MessagesState) -> Literal["tech_node", "billing_node"]:
            """Reads the last system message to determine the route."""
            messages = state["messages"]
            last_msg = messages[-1].content

            if "Billing" in last_msg:
                return "billing_node"
            return "tech_node"

        # Build the Parent Graph
        workflow = StateGraph(MessagesState)

        # Add the Nodes
        # Note: We can add the compiled agents directly as nodes!
        workflow.add_node("router", router_node)
        workflow.add_node("tech_node", tech_agent)
        workflow.add_node("billing_node", billing_agent)
        workflow.add_node("final_node", final_agent)

        # Entry point
        workflow.add_edge(START, "router")

        # Conditional Routing
        workflow.add_conditional_edges(
            "router",
            get_route,
            {
                "tech_node": "tech_node",
                "billing_node": "billing_node"
            }
        )

        # Fan-In: Both sub-agents go to the Final Agent
        workflow.add_edge("tech_node", "final_node")
        workflow.add_edge("billing_node", "final_node")

        # End
        workflow.add_edge("final_node", END)

        # Compile
        self.app = workflow.compile()

    def process(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
                
        response = self.app.invoke(
            {"messages": [{"role": "user", "content": prompt}]}
        )
        return {"text": str(response["messages"][-1].content)}
    