# === Third-party Library Imports ===
import dataiku

# == Dataiku-specific LLM integrations ==
from dataiku.llm.python import BaseLLM

# == LangChain / LangGraph Integrations ==
from langgraph.checkpoint.memory import InMemorySaver
from langgraph.prebuilt import create_react_agent
from langgraph_swarm import create_handoff_tool, create_swarm

# == Custom Tools ==
from tools import check_order_status, upgrade_subscription


class MyLLM(BaseLLM):
    def __init__(self):
        # Setup LLM
        LLM_ID = dataiku.get_custom_variables()["LLM_ID"]
        client = dataiku.api_client()
        project = client.get_default_project()
        llm = project.get_llm(LLM_ID).as_langchain_chat_model()
        
        # 2. Define Handoff Tools
        # These allow agents to transfer control explicitly
        transfer_to_shipping = create_handoff_tool(
            agent_name="Shipping", 
            description="Transfer to the shipping specialist for order tracking and delivery issues."
        )
        transfer_to_billing = create_handoff_tool(
            agent_name="Billing", 
            description="Transfer to the billing specialist for subscription upgrades and payments."
        )
        
        # 2. Define Agents
        # --- Shipping Agent ---
        # Goal: Fix shipping, then recognize if Billing is needed.
        shipping_agent = create_react_agent(
            llm,
            [check_order_status, transfer_to_billing],
            name="Shipping",
            prompt=(
                "You are the Shipping. Use tools to check order status."
                "Once you have answered the shipping question, REVIEW the user's original request."
                "If the user also asked about subscriptions/billing, transfer to Billing_Agent."
            ),
        )
        
        # --- Billing Agent ---
        # Goal: Fix billing, then close.
        billing_agent = create_react_agent(
            llm,
            [upgrade_subscription, transfer_to_shipping],
            name="Billing",
            prompt=(
                "You are the Billing. Help the user upgrade their plan."
                "Our Gold Tier is currently 20% off. Propose it."
            )
        )
        
        checkpointer = InMemorySaver()
        workflow = create_swarm(
            [shipping_agent, billing_agent],
            default_active_agent="Shipping"
        )
        self.app = workflow.compile(checkpointer=checkpointer)

   
    def process(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
        conv_id = "CONVERSATION_ID"
                
        config = {"configurable": {"thread_id": conv_id}}
        
        # Invoke the Agent
        response = self.app.invoke(
            {"messages": [{"role": "user", "content": prompt}]},
            config,
        )
        return {"text": str(response["messages"][-1].content)}
    