import dataiku
from dataiku.llm.python import BaseLLM
from dataikuapi.dss.llm import DSSLLMStreamedCompletionChunk, DSSLLMStreamedCompletionFooter
from typing import TypedDict, Annotated
from langchain_core.tools import StructuredTool
from langgraph.graph.message import add_messages
from langchain_core.messages import AnyMessage, HumanMessage, AIMessage
from langgraph.prebuilt import ToolNode
from langgraph.graph import START, StateGraph
from langgraph.prebuilt import tools_condition
from dataiku.langchain.dku_llm import DKULLM, DKUChatLLM
from langchain.tools import Tool

from langgraph.checkpoint.memory import InMemorySaver
from langgraph.prebuilt import create_react_agent
from langgraph_swarm import create_handoff_tool, create_swarm



# --- 1. Define Domain Tools ---

def check_order_status(order_id: str) -> str: 
    """Checks the status of a specific order ID."""
    # Mock database return
    return f"Order {order_id} is currently Out for Delivery. Expected arrival: Today by 5 PM."

def upgrade_subscription(tier: str) -> str:
    """Upgrades the user's plan to the specified tier."""
    return f"Success! The plan has been upgraded to the {tier} Tier."


class MyLLM(BaseLLM):
    def __init__(self):

        # Setup LLM
        LLM_ID = dataiku.get_custom_variables().get("LLM_ID", "your_llm_id")
        client = dataiku.api_client()
        project = client.get_default_project()
        llm = project.get_llm(LLM_ID).as_langchain_chat_model()
        
        # 2. Define Handoff Tools
        # These allow agents to transfer control explicitly
        transfer_to_shipping = create_handoff_tool(
            agent_name="Shipping", 
            description="Transfer to the shipping specialist for order tracking and delivery issues."
        )
        transfer_to_billing = create_handoff_tool(
            agent_name="Billing", 
            description="Transfer to the billing specialist for subscription upgrades and payments."
        )
        
        # --- Shipping Agent ---
        # Goal: Fix shipping, then recognize if Billing is needed.
        shipping_agent = create_react_agent(
            llm,
            [check_order_status, transfer_to_billing],
            name="Shipping",
            prompt=(
                "You are the Shipping. Use tools to check order status."
                "Once you have answered the shipping question, REVIEW the user's original request."
                "If the user also asked about subscriptions/billing, transfer to Billing_Agent."
            ),
        )
        
        # --- Billing Agent ---
        # Goal: Fix billing, then close.
        billing_agent = create_react_agent(
            llm,
            [upgrade_subscription, transfer_to_shipping],
            name="Billing",
            prompt=(
                "You are the Billing. Help the user upgrade their plan."
                "Our Gold Tier is currently 20% off. Propose it."
            )
        )
        
        
        checkpointer = InMemorySaver()
        workflow = create_swarm(
            [shipping_agent, billing_agent],
            default_active_agent="Shipping"
        )
        self.app = workflow.compile(checkpointer=checkpointer)

   
    def process(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
        print("PROMPT", prompt)
        print("QUERY", query)
        conv_id = 1#query["context"]["dku_conversation_id"]
                
        config = {"configurable": {"thread_id": conv_id}}

        response = self.app.invoke(
            {"messages": [{"role": "user", "content": prompt}]},
            config,
        )
        return {"text": str(response["messages"][-1].content)}
    