# === Third-party Library Imports ===
import dataiku
from typing import TypedDict, Annotated, List, Dict

# == Google ADK integrations ==
from google.adk.agents import LlmAgent
from google.adk.tools import FunctionTool, AgentTool
from google.adk.models.lite_llm import LiteLlm
from google.adk.artifacts import InMemoryArtifactService
from google.adk.runners import Runner
from google.adk.sessions import InMemorySessionService
from google.genai import types

# == Dataiku-specific LLM integrations ==
from dataiku.llm.python import BaseLLM

# == Custom tools from the Project Libraries ==
from tools import web_search

web_search_tool = FunctionTool(web_search)

class MyLLM(BaseLLM):
    def __init__(self):
        
        # Model name as recognized by *your* vLLM endpoint configuration
        project_variables = dataiku.get_custom_variables()
        LLM_ID = "openai/" + project_variables["LLM_ID"]
        
        # Endpoint URL provided by your vLLM deployment
        BASE_URL = "https://design.ds-platform.ondku.net/public/api/projects/MULTIAGENTSINDSS/llms/openai/v1/"

        # Credentials loading
        auth_info = dataiku.api_client().get_auth_info(with_secrets=True)
        for secret in auth_info["secrets"]:
            if secret["key"] == "api_key":
                API_KEY = secret["value"]
                print("API_KEY", API_KEY)
                break
                
        # Instantiate the LLM
        model = LiteLlm(
                model=LLM_ID,
                api_base=BASE_URL,
                api_key=API_KEY
        )
        
        # Define the sub-agents
        product_scout_agent = LlmAgent(
            name="product_scout_agent",
            model=model,
            tools=[web_search_tool],
            description="Summarize the specs and prices you find about a competitor",
            instruction="""
            You are a Product Researcher. 
            When asked about a competitor, use the Google Search Tool to find their official product pages, pricing pages, and tech blogs. 
            Summarize the specs and prices you find.
            """
        )
        
        marketing_intel_agent = LlmAgent(
            name="marketing_intel_agent",
            model=model,
            tools=[web_search_tool],
            description="Look for Announcing, New Campaign, or Commercial about a competitor",
            instruction="""
            You are a Marketing Researcher. 
            Use Google Search to find press releases, news articles, and blog posts about [Competitor]'s recent campaigns. 
            Look for Announcing, New Campaign, or Commercial.
            """
        )
            
        sentiment_analyst_agent = LlmAgent(
            name="sentiment_analyst_agent",
            model=model,
            tools=[web_search_tool],
            description="Search for public feedbacks on a competitor",
            instruction="""
            You are a Public Feedback Analyst. Use Google Search to find public sentiment. 
            You MUST use search operators to be effective.
            Example: site:reddit.com "Competitor X" complaints
            Example: site:trustpilot.com "Competitor X" Summarize the top 3 complaints and top 3 praises found in the search snippets.
            """
        )
        
        # Create parent agent and assign children via sub_agents
        self.coordinator = LlmAgent(
            name="coordinator",
            model=model,
            description="You are the coordinator agent.",
            instruction="""
# Role & Persona

You are the Supervisor Agent, orchestrating a competitive analysis report. You do not have direct internet access yourself; instead, you manage three specific "Worker Agents" (defined as tools below).

# Your Team (Worker Agents) 
You have access to four specialized agents. You must delegate tasks to them based on their specific skills:

1. Product_Scout_Agent: Finding facts about features, pricing, and specs.
2. Marketing_Intel_Agent: Finding press releases, news articles, and marketing announcements.
3. Sentiment_Analyst_Agent: Finding forum discussions and reviews.

# Operational Workflow (Chain of Thought)
When you receive a user request (e.g., "Generate a Q3 report on Competitor X"), you must follow this strict process:

Phase 1: Planning & Decomposition
- Analyze the request to identify the Target Competitor and the Timeframe (e.g., Q3, last 90 days).
- Create a mental outline for the final report (typically: Product, Marketing, Sentiment).

Phase 2: Delegation (Parallel Execution)
- Draft specific, actionable instructions for the three research agents.
- Instruction Rule: You must pass the specific Timeframe and Target Competitor to every agent.
  - To Product_Scout: Instruct them to find new features launched within the specific timeframe.
  - To Marketing_Intel: Instruct them to find campaigns run within the specific timeframe.
  - To Sentiment_Analyst: Instruct them to analyze sentiment over the specific timeframe.

Phase 3: Aggregation & Synthesis
- Wait for the outputs from the three research agents.
- Do not summarize the data yourself yet. Gather the raw outputs into a structured context block.

Phase 4: Final Execution
Compile this data into the following Markdown Report Template. You must strictly map the output of each Worker Agent to its corresponding section below.

REPORT TEMPLATE START

Q3 Competitive Intelligence Report: [Target Competitor Name]
Reporting Period: Q3 (July 1 – September 30) Date Generated: [Current Date]

1. Executive Summary
(Instruction: Synthesize the combined findings from all three agents into a 3-sentence overview. Focus on the most significant shift observed in Q3.)

The Q3 Bottom Line: [Insert Summary Here]

2. Product Intelligence (Source: Product_Scout_Agent)
If the agent returned "No data found," explicitly state "No major product updates detected in Q3."

New Feature Releases & Updates
[Feature Name/Update]: [Description of what changed]

Impact: (Is this a minor tweak or a major capability?)

[Feature Name/Update]: [Description]

Pricing & Packaging Changes
Pricing Structure: [Note any price increases, decreases, or discount changes observed in Q3]

Trial/Freemium Models: [Note changes to free tier allowances]

3. Marketing Strategy (Source: Marketing_Intel_Agent)
If the agent returned "No data found," explicitly state "No major campaigns detected in Q3."

Key Q3 Campaigns & Press
Campaign Theme: [e.g., "Back to School Push" or "Summer Efficiency"]

Major Announcements: [List partnerships, funding news, or acquisitions]

Messaging Shift
Observation: How did their tagline or value proposition change from the start of July to the end of September?

[Insert Analysis]

4. Market Sentiment (Source: Sentiment_Analyst_Agent)
If the agent returned "No data found," explicitly state "Insufficient review volume for analysis."

The "Voice of the Customer"
Dominant Positive Theme: [What are users praising most in Q3?]

Dominant Negative Theme: [What are users complaining about most in Q3?]

Verbatim Evidence
"[Insert a direct quote or summary of a review that represents the general Q3 sentiment]"

5. Strategic Synthesis (Supervisor Analysis)
(Instruction: Based on the data above, provide the following specific outputs for our internal team.)

Threat Level (Low/Medium/High): [Assessment]

Primary Q4 Prediction: Based on their Q3 moves, what do we expect them to do in Q4 (October-December)?

Counter-Strategy: Recommend one specific action our team should take to counter their Q3 progress.

REPORT TEMPLATE END

# Constraints
- Do not invent data. If a worker agent returns "No data found," reflect that in the final report.
- Do not perform the analysis yourself. Always delegate.
- Maintain a professional, objective business tone in your coordination.
        """,
            sub_agents=[
                product_scout_agent,
                marketing_intel_agent,
                sentiment_analyst_agent
            ],
            tools=[
                AgentTool(product_scout_agent),
                AgentTool(marketing_intel_agent),
                AgentTool(sentiment_analyst_agent)
            ]
        )

    async def call_agent_async(self, query: str, runner, user_id, session_id):
        """Sends a query to the agent and prints the final response."""

        # Prepare the user's message in ADK format
        content = types.Content(role='user', parts=[types.Part(text=query)])

        final_response_text = "Agent did not produce a final response." # Default

        # Key Concept: run_async executes the agent logic and yields Events.
        # We iterate through events to find the final answer.
        async for event in runner.run_async(user_id=user_id, session_id=session_id, new_message=content):

          # Key Concept: is_final_response() marks the concluding message for the turn.
          if event.is_final_response():
            if event.content and event.content.parts:
                # Assuming text response in the first part
                final_response_text = event.content.parts[0].text
            elif event.actions and event.actions.escalate: # Handle potential errors/escalations
                final_response_text = f"Agent escalated: {event.error_message or 'No specific message.'}"
            # Add more checks here if needed (e.g., specific error codes)
            break # Stop processing events once the final response is found

        return final_response_text

    async def run_team_conversation(self, query):
        APP_NAME="APPI_NAME"
        USER_ID="USER_ID"
        SESSION_ID="SESSION_ID"
        session_service = InMemorySessionService()
        session = await session_service.create_session(app_name=APP_NAME, user_id=USER_ID, session_id=SESSION_ID)
        print(f"Session created: App='{APP_NAME}', User='{USER_ID}', Session='{SESSION_ID}'")

        runner_agent_team = Runner(agent=self.coordinator, app_name=APP_NAME, session_service=session_service)
        print(f"Runner created for agent '{self.coordinator.name}'.")

        final_response = await self.call_agent_async(query=query,
                                               runner=runner_agent_team,
                                               user_id=USER_ID,
                                               session_id=SESSION_ID
                                               )
        
        return final_response

    async def aprocess(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
        
        # Invoke the LLM
        resp_text = await self.run_team_conversation(prompt)

        return {"text": resp_text}
    