# === Third-party Library Imports ===
import dataiku

# == LangGraph / LangChain Integrations ==
from langgraph.prebuilt import create_react_agent
from langgraph_supervisor import create_supervisor

# == Dataiku LLM Integration ==
from dataiku.llm.python import BaseLLM

# == Custom Tools ==
from tools import web_search


class MyLLM(BaseLLM):
    def __init__(self):
        
        # Instantiate the LLM
        LLM_ID = dataiku.get_custom_variables()["LLM_ID"]
        client = dataiku.api_client()
        project = client.get_default_project()
        llm = project.get_llm(LLM_ID).as_langchain_chat_model()
        
        # Define the sub-agents
        product_scout_agent = create_react_agent(
            model=llm,
            tools=[web_search],
            name="product_scout_agent",
            prompt=("You are a Product Researcher. When asked about a competitor, use the Google Search Tool to find the product information. Summarize your findings as a string. Use the tool only once.")
        )
        
        marketing_intel_agent = create_react_agent(
            model=llm,
            tools=[web_search],
            name="marketing_intel_agent",
            prompt=("You are a Marketing Researcher. Use Google Search to find news information about [Competitor]'s recent campaigns. Look for Announcing, New Campaign, or Commercial. Use the tool only once.")
        )
            
        sentiment_analyst_agent = create_react_agent(
            model=llm,
            tools=[web_search],
            name="sentiment_analyst_agent",
            prompt=("""You are a Public Feedback Analyst. Use Google Search to do one search to find public sentiment. Use the tool only once for the most relevant site.
                """)
        )
        
        # Define the supervisor
        workflow = create_supervisor(
            [product_scout_agent, marketing_intel_agent, sentiment_analyst_agent],
            model=llm,
            prompt="""
            You are the Supervisor Agent orchestrating a competitive analysis report.

            WORKFLOW:
            1. Use the once each expert tool to gather data.
            2. Synthesize findings into the final Markdown Report Template.

            REPORT TEMPLATE:
            # Q3 Competitive Intelligence Report: [Target]
            1. Executive Summary: [3-sentence overview]
            2. Product Intelligence: [1 short paragraph - Data from Product Scout]
            3. Marketing Strategy: [1 short paragraphData from Marketing Intel]
            4. Market Sentiment: [1 short paragraphData from Sentiment Analyst]
            5. Strategic Synthesis: [Your prediction for Q4]
            """
        )
        
        self.app = workflow.compile()
   
    def process(self, query, settings, trace):
        
        # Invoke the agent
        response = self.app.invoke({"messages": query["messages"]})
     
        return {"text": response['messages'][-1].content}
    