# === 1. Standard & Third-party Imports ===
from openai import AsyncOpenAI

# === 2. Dataiku & Agents Framework ===
import dataiku
from dataiku.llm.python import BaseLLM

# === 3. OpenAI Agent SDK Framework === 
from agents import Agent, Runner, function_tool, OpenAIChatCompletionsModel

@function_tool
def web_search(search_query: str) -> str:
    """
    Performs a Google search to find product specs, pricing, news, or sentiment.
    """
    web_search_tool = dataiku.api_client().get_default_project().get_agent_tool("K6yKQTA")
    output = web_search_tool.run({"q": search_query})
    return str(output["output"])

class MyLLM(BaseLLM):
    def __init__(self):
        # 1. Setup Model
        LLM_ID = dataiku.get_custom_variables()["LLM_ID"]
        BASE_URL = "https://design.ds-platform.ondku.net/public/api/projects/MULTIAGENTSINDSS/llms/openai/v1/"
        auth_info = dataiku.api_client().get_auth_info(with_secrets=True)
        API_KEY = next(s["value"] for s in auth_info["secrets"] if s["key"] == "api_key")
        
        openai_client = AsyncOpenAI(base_url=BASE_URL, api_key=API_KEY)
        self.model = OpenAIChatCompletionsModel(model=LLM_ID, openai_client=openai_client)

        # 2. Define sub-agents
        product_scout = Agent(
            name="product_scout_agent",
            instructions="You are a Product Researcher. Find official specs and pricing by using the search tool available only once.",
            model=self.model,
            tools=[web_search]
        )

        marketing_intel = Agent(
            name="marketing_intel_agent",
            instructions="You are a Marketing Researcher. Find press releases and campaignsby using the search tool available only once.",
            model=self.model,
            tools=[web_search]
        )

        sentiment_analyst = Agent(
            name="sentiment_analyst_agent",
            instructions="You are a Feedback Analyst. Use site:reddit.com or site:trustpilot.comby using the search tool available only once.",
            model=self.model,
            tools=[web_search]
        )

        # 3. Define the Supervisor as a "Manager"
        # We use .as_tool() to expose the sub-agents to the Supervisor
        self.supervisor = Agent(
            name="supervisor_agent",
            model=self.model,
            instructions="""
            You are the Supervisor Agent orchestrating a competitive analysis report.
            
            WORKFLOW:
            1. Use the once each expert tool to gather data.
            2. Synthesize findings into the final Markdown Report Template.
            
            REPORT TEMPLATE:
            # Q3 Competitive Intelligence Report: [Target]
            1. Executive Summary: [3-sentence overview]
            2. Product Intelligence: [1 short paragraph - Data from Product Scout]
            3. Marketing Strategy: [1 short paragraphData from Marketing Intel]
            4. Market Sentiment: [1 short paragraphData from Sentiment Analyst]
            5. Strategic Synthesis: [Your prediction for Q4]
            """,
            tools=[
                product_scout.as_tool(
                    tool_name="product_expert",
                    tool_description="Retrieves feature updates and pricing data."
                ),
                 marketing_intel.as_tool(
                     tool_name="marketing_expert",
                     tool_description="Retrieves news and campaign announcements."
                 ),
                 sentiment_analyst.as_tool(
                     tool_name="sentiment_expert",
                     tool_description="Retrieves public reviews and sentiment."
                 )
            ]
        )

    async def aprocess(self, query, settings, trace):
        prompt = query["messages"][-1]["content"]
        
        # Invoke the agent
        result = await Runner.run(self.supervisor, prompt, max_turns=15)
        return {"text": result.final_output}
    