import dataiku
import random
import re
import numpy as np

customers_df = (
    dataiku.Dataset("customers_info_sql")
    .get_dataframe()
    .replace(np.nan, "None", regex=True)
    .set_index("id")
)


def extract_integer(s):
    match = re.search(r"\d+", s)
    if match:
        return int(match.group())
    else:
        return None


def get_customer_id(name):
    """
    Provide the customer id, given the customer's name.
    This is the first thing to do if the customer's id is unknown.
    The input should be a a string like 'Smith' or 'Jane Snow' (without titles like "Mr.", "Mrs."...)
    """
    if len(name) == 0:
        return "No customer with this name. A name should be provided, e.g. 'Smith' or 'Jane Snow'"

    words = ("".join(x if x.isalnum() else " " for x in name)).lower().split()
    mask = customers_df.apply(
        lambda row: all(
            w in row["first_name"].lower() or w in row["last_name"].lower()
            for w in words
        ),
        axis=1,
    )
    if sum(mask) == 0:
        return "No customer with this name. A name should be provided, e.g. 'Smith' or 'Jane Snow'"
    result = """\n| customer | id |
    |---|---|"""

    masked = customers_df[mask]
    for row in masked.iterrows():
        result += f"\n| {row[1].first_name} {row[1].last_name} | {row[0]} |"
    if len(masked) > 1:
        result += "\nThere is more than 1 result. Please be more specific"
    return result


def get_details(customer_id):
    """
    Provide the details of a given customer (identity, contact details, subscription date, options).
    The input should be a single integer representing the customer's id. E.g. '123'.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        name = f"{row.title} {row.first_name} {row.last_name}"
        return f"{name}, address: {row.address}, email: {row.email_address}, subscription date: {row.subscription_date}, options: {row.options}"
    except (KeyError, ValueError):
        return "Unknown customer id"


def reset_password(customer_id):
    """
    Reset the password of a customer.
    Use this if the customer mentions problems with his or her password.
    The input should be a single integer representing the customer's id. E.g. '123'.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        return f"Email sent so that customer #{customer_id} can define a new password. No further action is needed to reinitialize the password."
    except (KeyError, ValueError):
        return "Unknown customer id"


def schedule_local_intervention(customer_id):
    """
    Send an email allowing the customer to schedule an appointment for a technician's visit.
    The input should be a single integer representing the customer's id. E.g. '123'.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        return f"Email sent to customer #{customer_id} to plan the visit of a technician. No further action is needed to handle an Internet connection problem."
    except (KeyError, ValueError):
        return "Unknown customer id"


def schedule_distant_intervention(customer_id):
    """
    Send an email allowing the customer to schedule an appointment for a a phone discussion with a technician.
    The input should be a single integer representing the customer's id. E.g. '123'.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        return f"Email sent to customer #{customer_id} to plan a phone discussion with a technician. No further action is needed to handle an Internet connection problem."
    except (KeyError, ValueError):
        return "Unknown customer id"


def cancel_appointment(customer_id):
    """
    Cancel an upcoming appointment. An appointment needs to be canceled before planning a new appointment.
    The input should be a single integer representing the customer's id. E.g. '123'.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        return f"Upcoming appointment canceled for customer #{customer_id}"
    except (KeyError, ValueError):
        return "Unknown customer id"


def sign_up_to_option(inputs):
    """
    Activate an option for the customer's Internet plan.
    Only use this action if the customer explicitly requested a new option.
    Use this action once for each option.
    The two available options are 'TV' and 'Premium'.
    The input should be a single integer for the customer's id followed by a comma then a either 'TV' or 'Premium'
    """
    try:
        splitted = inputs.split(",")
        customer_id = extract_integer(splitted[0])
        option = ("".join(c for c in splitted[1] if c.isalpha())).lower()
        if option in ["tv", "premium"]:
            option_label = "TV" if option == "tv" else "Premium"
            if option_label not in customers_df.loc[customer_id, "options"]:
                return f"The {option_label} option has successfully been activated for customer #{customer_id}"
            else:
                return f"The {option_label} option was already activated for customer #{customer_id}"
        else:
            return "Incorrect input. The input for the option should be either 'TV' or 'Premium'"
    except (KeyError, ValueError, IndexError):
        pass
    return "Incorrect input. The input should be a single integer for the customer's id followed by a comma then either 'TV' or 'Premium'"


def run_diagnostics(customer_id):
    """
    Determine the best course of action when a customer experiences an Internet connection problem.
    This should be the first action to take if a customer reports an Internet connection problem.
    The input should be a single integer representing the customer's id. E.g. '123'.
    """

    try:
        row = customers_df.loc[customer_id]
        situation = customer_id % 3
        if situation == 0:
            return f"The customer can directly solve the connection problem by following the instructions provided in the FAQ: www.telco-operator/faq/{random.randint(1, 100)}. An intervention of a technician is not warranted."
        elif situation == 1:
            return f"A technician must visit the customer to solve the connection problem. An appointment needs to be made"
        else:
            return f"A technician must discuss with the customer over the phone to solve the connection problem. An appointment needs to be made"
    except (KeyError, ValueError):
        return "Unknown customer id"


def cancel_option(inputs):
    """
    Dectivate an option for the customer's Internet plan.
    Only use this action if the customer explicitly requested a new option.
    Use this action once for each option.
    The two available options are 'TV' and 'Premium'.
    The input should be a single integer for the customer's id followed by a comma then a either 'TV' or 'Premium'
    """
    try:
        splitted = inputs.split(",")
        customer_id = extract_integer(splitted[0])
        option = ("".join(c for c in splitted[1] if c.isalpha())).lower()
        if option in ["tv", "premium"]:
            option_label = "TV" if option == "tv" else "Premium"
            if option_label in customers_df.loc[customer_id, "options"]:
                return f"The {option_label} option has successfully been deactivated for customer #{customer_id}"
            else:
                return f"The {option_label} option was not activated for customer #{customer_id}"
        else:
            return "Incorrect input. The input for the option should be either 'TV' or 'Premium'"
    except (KeyError, ValueError, IndexError):
        pass
    return "Incorrect input. The input should be a single integer for the customer's id followed by a comma then either 'TV' or 'Premium'"


def cancel_phone_subscription(customer_id):
    """
    Cancel the customer's phone subscription.
    Only used this if necessary or requested by the customer.
    Before doing this, check in the manual whether you have the right to do so.

    The input should be a single integer for the customer's id.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        return f"The phone subscription of customer #{customer_id} has successfully been canceled."
    except:
        pass
    return f"The customer #{customer_id} is not present in the database."


def get_product_info(input):
    """
    Provide the characteristics of a product. Input is the name of the product and output is the description. The only products available are:
    - "iPhone 10"
    - "iPhone 11"
    - "Samsung galaxy S9"
    - "Samsung galaxy S10"
    """
    input = input.replace('"', "").replace("'", "").lower()

    if input == "iphone 10":
        return "The iPhone 10 costs 300 dollars, it is available and can be sent in 3 days."
    elif input == "iphone 11":
        return "The iPhone 11 costs 600 dollars, it is available and can be sent in one week"
    elif input == "samsung galaxy s9":
        return "The Samsung galaxy S9 costs 200 dollars, it is available and can be sent in 3 days."
    elif input == "samsung galaxy s10":
        return "The Samsung galaxy S10 costs 250 dollars, it is available and can be sent in 3 days."
    return """There is no information for the specific product name you entered, the only entries available are ["Iphone 10", "Iphone 11", "Samsung galaxy S9", "Samsung galaxy S10"]"""


def subscribe_to_loyalty_program(customer_id):
    """
    Enroll a user to the customer loyalty program.
    Check in the manual first to see if the customer has access to this subscription.

    The input should be a single integer for the customer's id.
    """
    try:
        row = customers_df.loc[extract_integer(customer_id)]
        return f"The customer #{customer_id} has successfully subscribed to the customer loyalty program."
    except:
        pass
    return f"The customer #{customer_id} does not seem to be in the database."
