# Scenarios[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#scenarios "Permalink to this heading")

The scenario API can control all aspects of managing a scenario.

Note

There is a dedicated API to use *within* a scenario to run steps and report on progress of the scenario. For that, please see Scenarios (in a scenario).

## Example use cases[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#example-use-cases "Permalink to this heading")

In all examples, `project` is a `dataikuapi.dss.project.DSSProject` handle, obtained using `client.get\_project()` or `client.get\_default\_project()`

### Run a scenario[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#run-a-scenario "Permalink to this heading")

#### Variant 1: Run and wait for it to complete[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#variant-1-run-and-wait-for-it-to-complete "Permalink to this heading")

§ scenario = project.get\_scenario("myscenario")

§ scenario.run\_and\_wait()

#### Variant 2: Run, then poll while doing other stuff[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#variant-2-run-then-poll-while-doing-other-stuff "Permalink to this heading")

§ scenario = project.get\_scenario("myscenario")

§ trigger\_fire = scenario.run()

§ # When you call `run` a scenario, the scenario is not immediately

§ # started. Instead a "manual trigger" fires.

§ #

§ # This trigger fire can be cancelled if the scenario was already running,

§ # or if another trigger fires

§ # Thus, the scenario run is not available immediately, and we must "wait"

§ # for it

§ scenario\_run = trigger\_fire.wait\_for\_scenario\_run()

§ # Now the scenario is running. We can wait for it synchronously with

§ # scenario\_run.wait\_for\_completion(), but if we want to do other stuff

§ # at the same time, we can use refresh

§ while True:

§ # Do a bit of other stuff

§ # ...

§ scenario\_run.refresh()

§ if scenario\_run.running:

§ print("Scenario is still running ...")

§ else:

§ print("Scenario is not running anymore")

§ break

§ time.sleep(5)

### Get information about the last completed run of a scenario[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#get-information-about-the-last-completed-run-of-a-scenario "Permalink to this heading")

§ scenario = project.get\_scenario("myscenario")

§ last\_runs = scenario.get\_last\_runs(only\_finished\_runs=True)

§ if len(last\_runs) == 0:

§ raise Exception("The scenario never ran")

§ last\_run = last\_runs[0]

§ # outcome can be one of SUCCESS, WARNING, FAILED or ABORTED

§ print("The last run finished with %s" % last\_run.outcome)

§ # start\_time and end\_time are datetime.datetime objects

§ print("Last run started at %s and finished at %s" % (last\_run.start\_time, last\_run.end\_time))

### Disable/enable scenarios[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#disable-enable-scenarios "Permalink to this heading")

#### Disable and remember[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#disable-and-remember "Permalink to this heading")

This snippet disables all scenarios in a project (i.e. prevents them from auto-triggering), and also keeps a list of the ones that were active, so that you can selectively re-enable them later

§ # List of scenario ids that were active

§ previously\_active = []

§ for scenario in project.list\_scenarios(as\_type="objects"):

§ settings = scenario.get\_settings()

§ if settings.active:

§ previously\_active.append(scenario.id)

§ settings.active = False

§ # In order for settings change to take effect, you need to save them

§ settings.save()

#### Enable scenarios from a list of ids[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#enable-scenarios-from-a-list-of-ids "Permalink to this heading")

§ for scenario\_id in previously\_active:

§ scenario = project.get\_scenario(scenario\_id)

§ settings = scenario.get\_settings()

§ settings.active = True

§ settings.save()

### List the “run as” user for all scenarios[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#list-the-run-as-user-for-all-scenarios "Permalink to this heading")

This snippet allows you to list the identity under which a scenario runs:

§ for scenario in project.list\_scenarios(as\_type="objects"):

§ settings = scenario.get\_settings()

§ # We must use `effective\_run\_as` and not `run\_as` here.

§ # run\_as contains the "configured" run as, which can be None - in that case, it will run

§ # as the last modifier of the scenario

§ # effective\_run\_as is always valued and is the resolved version.

§ print("Scenario %s runs as user %s" % (scenario.id, settings.effective\_run\_as))

### Reassign scenarios to another user[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#reassign-scenarios-to-another-user "Permalink to this heading")

If user “u1” has left the company, you may want to reassign all scenarios that ran under his identity to another user “u2”.

§ for scenario in project.list\_scenarios(as\_type="objects"):

§ settings = scenario.get\_settings()

§ if settings.effective\_run\_as == "u1":

§ print("Scenario %s used to run as u1, reassigning it")

§ # To configure a run\_as, we must use the run\_as property.

§ # effective\_run\_as is read-only

§ settings.run\_as = "u2"

§ settings.save()

### Get the “next expected” run for a scenario[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#get-the-next-expected-run-for-a-scenario "Permalink to this heading")

If the scenario has a temporal trigger enabled, this will return a datetime of the approximate next expected run

§ scenario = project.get\_scenario("myscenario")

§ # next\_run is None if no next run is scheduled

§ print("Next run is at %s" % scenario.get\_status().next\_run)

### Get the list of jobs started by a scenario[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#get-the-list-of-jobs-started-by-a-scenario "Permalink to this heading")

“Build/Train” or Python steps in a scenario can start jobs. This snippet will give you the list of job ids that a particular scenario run executed.

These job ids can then be used together with `dataikuapi.dss.project.DSSProject.get\_job()`

§ scenario = project.get\_scenario("myscenario")

§ # Focusing only on the last completed run. Else, use get\_last\_runs() and iterate

§ last\_run = scenario.get\_last\_finished\_run()

§ last\_run\_details = last\_run.get\_details()

§ all\_job\_ids = []

§ for step in last\_run\_details.steps:

§ all\_job\_ids.extend(step.job\_ids)

§ print("All job ids started by scenario run %s : %s" % (last\_run.id, all\_job\_ids))

### Get the first error that happened in a scenario run[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#get-the-first-error-that-happened-in-a-scenario-run "Permalink to this heading")

This snippet retrieves the first error that happened during a scenario run.

§ scenario = project.get\_scenario("myscenario")

§ last\_run = scenario.get\_last\_finished\_run()

§ if last\_run.outcome == "FAILED":

§ last\_run\_details = last\_run.get\_details()

§ print("Error was: %s" % (last\_run\_details.first\_error\_details))

### Start multiple scenarios and wait for all of them to complete[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#start-multiple-scenarios-and-wait-for-all-of-them-to-complete "Permalink to this heading")

This code snippet starts multiple scenarios and returns when all of them have completed, returning the updated DSSScenarioRun for each

§ import time

§ scenarios\_ids\_to\_run = ["s1", "s2", "s3"]

§ scenario\_runs = []

§ for scenario\_id in scenarios\_ids\_to\_run:

§ scenario = project.get\_scenario(scenario\_id)

§ trigger\_fire = scenario.run()

§ # Wait for the trigger fire to have actually started a scenario

§ scenario\_run = trigger\_fire.wait\_for\_scenario\_run()

§ scenario\_runs.append(scenario\_run)

§ # Poll all scenario runs, until all of them have completed

§ while True:

§ any\_not\_complete = False

§ for scenario\_run in scenario\_runs:

§ # Update the status from the DSS API

§ scenario\_run.refresh()

§ if scenario\_run.running:

§ any\_not\_complete = True

§ if any\_not\_complete:

§ print("At least a scenario is still running...")

§ else:

§ print("All scenarios are complete")

§ break

§ # Wait a bit before checking again

§ time.sleep(30)

§ print("Scenario run ids and outcomes: %s" % ([(sr.id, sr.outcome) for sr in scenario\_runs]))

### Change the “from” email for email reporters[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#change-the-from-email-for-email-reporters "Permalink to this heading")

Note that usually, we would recommend using variables for “from” and “to” email. But you can also modify them with the API.

§ scenario = project.get\_scenario("myscenario")

§ settings = scenario.get\_settings()

§ for reporter in settings.raw\_reporters:

§ # Only look into 'email' kind of reporters

§ if reporter["messaging"]["type"] == "mail-scenario":

§ messaging\_configuration = reporter["messaging"]["configuration"]

§ messaging\_configuration["sender"] = "new.email.address@company.com"

§ print("Updated reporter %s" % reporter["id"])

§ settings.save()

## Detailed examples[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#detailed-examples "Permalink to this heading")

This section contains more advanced examples on Scenarios.

### Get last run results[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#get-last-run-results "Permalink to this heading")

You can programmatically get the outcome of the last finished run for a given Scenario.

§ scenario = project.get\_scenario(scenario\_id)

§ last\_run = scenario.get\_last\_finished\_run()

§ data = {

§ "scenario\_id": scenario\_id,

§ "outcome": last\_run.outcome,

§ "start\_time": last\_run.start\_time.isoformat(),

§ "end\_time": last\_run.end\_time.isoformat()

§ }

§ print(data)

From there, you can easily extend the same logic to loop across all Scenarios within a Project.

### Define Scenario timeout[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#define-scenario-timeout "Permalink to this heading")

There is no explicit timeout functionality for “Build” steps within Dataiku Scenarios, however in the case of custom Scenarios you can implement one using the public API’s Python client. The following example is a custom Scenario with a `build\_dataset` step which is aborted if it takes more than 1 hour (3600 seconds) to run.

§ ###########################################################################################

§ # !! CUSTOM SCENARIO EXAMPLE !! #

§ # See https://doc.dataiku.com/dss/latest/scenarios/custom\_scenarios.html for more details #

§ ###########################################################################################

§ import time

§ import dataiku

§ from dataiku.scenario import Scenario, BuildFlowItemsStepDefHelper

§ from dataikuapi.dss.future import DSSFuture

§ TIMEOUT\_SECONDS = 3600

§ s = Scenario()

§ # Replace this commented block by your Scenario steps

§ # Example: build a Dataset

§ step\_handle = s.build\_dataset("your\_dataset\_name", asynchronous=True)

§ start = time.time()

§ while not step\_handle.is\_done():

§ end = time.time()

§ print("Duration: {}s".format(end-start))

§ if end - start > TIMEOUT\_SECONDS:

§ f = DSSFuture(dataiku.api\_client(), step\_handle.future\_id)

§ f.abort()

§ raise Exception("Scenario was aborted because it took too much time.")

## Reference documentation[¶](https://developer.dataiku.com/latest/concepts-and-examples/scenarios.html#reference-documentation "Permalink to this heading")

|  |  |

| --- | --- |

| `dataikuapi.dss.scenario.DSSScenario`(client, ...) | A handle to interact with a scenario on the DSS instance. |

| `dataikuapi.dss.scenario.DSSScenarioSettings`(...) | Settings of a scenario. |

| `dataikuapi.dss.scenario.DSSScenarioRun`(...) | A handle containing basic info about a past run of a scenario. |

| `dataikuapi.dss.scenario.DSSTriggerFire`(...) | A handle representing the firing of a trigger on a scenario. |
