# How to programmatically set email recipients in a “Send email” reporter using the API?[¶](https://knowledge.dataiku.com/latest/kb/o16n/automation/How-to-programmatically-set-email-recipients-in-a-reporter.html#how-to-programmatically-set-email-recipients-in-a-send-email-reporter-using-the-api "Permalink to this headline")

The Dataiku DSS API allows the users to programmatically interact with the product. It can be very useful when having to apply some operations that would be quite repetitive to achieve through the UI, or when it comes to automating some interactions with DSS.

In this use case, let’s consider we have a project containing multiple scenarios, and for all of them, we want to add a new recipient for all the existing “Send email” reporters.

We’re going to achieve that with a python script that will be executed from inside of DSS, but the same logic can be used from outside of DSS.

The idea of this operation is to list all the existing scenarios in a specified project then search for all the “Send email” reporters, then retrieve the list of recipients and then finally update the list if the new recipient doesn’t already exist.

To interact with scenarios, we first need to access the REST API client. From a script running inside of DSS, it’s pretty straight forward using `dataiku.api\_client()`:

§ import dataiku

§ client = dataiku.api\_client()

From outside of DSS, you need additional steps to configure how to access the instance. Please refer to the article Using the APIs outside of DSS from the reference documentation to know more.

Then, let’s create a variable to store the new recipient email address:

§ new\_recipient\_email\_address = "john.doe@here.com"

The next step is to retrieve the project and the list of scenario it contains:

§ project = client.get\_project("PROJECT\_KEY")

§ scenarios\_list = project.list\_scenarios()

`list\_scenarios()` returns a dictionary, let’s use the id property to retrieve a handle to interact with the scenario:

§ for scenario\_metadata in scenarios\_list:

§ scenario = project.get\_scenario(scenario\_metadata['id'])

Let’s then retrieve the scenario definition to interact with the scenario attributes:

§ scenario\_definition = scenario.get\_definition(with\_status=False)

Now, it’s time to iterate on all the existing reporters, check if they are “Send email” reporters, if so then retrieve the existing list of recipients, and add the new recipient email address when missing:

§ update\_scenario = False

§ for i in range(0, len(scenario\_definition['reporters'])):

§ if scenario\_definition['reporters'][i]['messaging']['type'] == "mail-scenario":

§ recipients = [recipient.strip() for recipient in scenario\_definition['reporters'][i]['messaging']['configuration']['recipient'].split(',')]

§ if not new\_recipient\_email\_address in recipients:

§ recipients.append(new\_recipient\_email\_address)

§ scenario\_definition['reporters'][i]['messaging']['configuration']['recipient'] = ', '.join(recipients)

§ update\_scenario = True

§ print("Updating recipient for mail reporter \"{}\" of scenario \"{}\"".format(scenario\_definition['reporters'][i]['name'], scenario\_metadata['name']))

Finally, if we’ve edited the list of recipients, let’s update the definition of the scenario :

§ if update\_scenario:

§ scenario.set\_definition(scenario\_definition,with\_status=False)

## Final code sample[¶](https://knowledge.dataiku.com/latest/kb/o16n/automation/How-to-programmatically-set-email-recipients-in-a-reporter.html#final-code-sample "Permalink to this headline")

§ import dataiku

§ client = dataiku.api\_client()

§ project = client.get\_project("PROJECT\_KEY")

§ scenarios\_list = project.list\_scenarios()

§ new\_recipient\_email\_address = "john.doe@here.com"

§ for scenario\_metadata in scenarios\_list:

§ scenario = project.get\_scenario(scenario\_metadata['id'])

§ scenario\_definition = scenario.get\_definition(with\_status=False)

§ update\_scenario = False

§ for i in range(0, len(scenario\_definition['reporters'])):

§ if scenario\_definition['reporters'][i]['messaging']['type'] == "mail-scenario":

§ recipients = [recipient.strip() for recipient in scenario\_definition['reporters'][i]['messaging']['configuration']['recipient'].split(',')]

§ if not new\_recipient\_email\_address in recipients:

§ recipients.append(new\_recipient\_email\_address)

§ scenario\_definition['reporters'][i]['messaging']['configuration']['recipient'] = ', '.join(recipients)

§ update\_scenario = True

§ print("Updating recipient for mail reporter \"{}\" of scenario \"{}\"".format(scenario\_definition['reporters'][i]['name'], scenario\_metadata['name']))

§ if update\_scenario:

§ scenario.set\_definition(scenario\_definition,with\_status=False)

This code sample is also available on our GitHub repository.

## What’s next?[¶](https://knowledge.dataiku.com/latest/kb/o16n/automation/How-to-programmatically-set-email-recipients-in-a-reporter.html#what-s-next "Permalink to this headline")

* For more details about how to use the API, see Python APIs

* For more details about how to interact with projects using the python API, see Managing projects

* For more details about how to interact with scenarios using the python API, see Managing scenarios
