import time 
import logging
from flask import Blueprint, jsonify, request, Response
from typing import Dict
from langchain_core.messages import HumanMessage
from datetime import datetime

from ..agentic_insights_toolkit.dku_utils.object import find_agent
from ..agentic_insights_toolkit.dku_utils.utils import list_zone_datasets, get_dataset_columns
from ..agentic_insights_toolkit.utils.analysis_session import (
    get_all_analysis_session_metadata,
    generate_analysis_session_id,
    get_analysis_report,
    get_analysis_session_metadata,
    update_analysis_session_metadata,
    delete_analysis_session,
    get_available_analytic_tools,
    get_main_analysis_session_scenario_run_status,
    abort_main_analysis_session_scenario,
    update_analysis_session_params_in_flow,
    get_report_assistant_webapp_relative_path
)
from .utils import convert_nan_to_none, start_new_analysis_session

analysis_session_api = Blueprint("analysis_session_api", __name__, url_prefix="/api")

@analysis_session_api.route("/generate-analysis-session-id", methods=["GET"])
def generate_analysis_session_id_api():
    """
    Generate a new analysis session ID.
    """
    try:
        analysis_session_id = generate_analysis_session_id()
        return jsonify({"ok": True, "analysis_session_id": analysis_session_id})
    except Exception as e:
        logging.exception("Error generating analysis session ID")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/start-analysis-session", methods=["POST"])
def start_analysis_session_api():
    """
    Start a new analysis session with SSE streaming.
    """
    try:
        data = request.get_json()
        if not data:
            return jsonify({"ok": False, "error": "No data received"}), 400
            
        session_params = {
            "analysis_session_id": data.get("analysis_session_id") or "",
            "input_datasets": data.get("input_datasets") or [],
            "analysis_to_perform": data.get("analysis_to_perform") or [],
            "infer_target_variable": data.get("infer_target_variable", False),
            "manual_target_col_for_rca": data.get("manual_target_col_for_rca") or "",
            "use_business_rules": data.get("use_business_rules", True),
            "additional_context": data.get("additional_context") or "",
            "report_template_id": data.get("report_template_id") or "",
            "start_datetime": data.get("start_datetime") or "",
            "end_datetime": data.get("end_datetime") or ""
        }

       
        update_analysis_session_metadata(analysis_session_id=session_params["analysis_session_id"], metadata=session_params)
    
        scenario_run_insights = start_new_analysis_session(session_params = session_params)
    
        # Update end_datetime and status in metadata
        session_params["end_datetime"] = datetime.utcnow().isoformat() + "Z"
        session_params["status"] = scenario_run_insights.get("status")
        update_analysis_session_metadata(analysis_session_id=session_params["analysis_session_id"], metadata=session_params)

        update_analysis_session_params_in_flow(analysis_session_id=session_params["analysis_session_id"])

        return jsonify({"ok": True, "scenario_run_insights": scenario_run_insights})
        

    except Exception as e:
        logging.exception("Error starting analysis session")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/abort-analysis-session", methods=["POST"])
def abort_analysis_session_api():
    """
    Abort an analysis session.
    """
    try:
        data = request.get_json()
        abort_main_analysis_session_scenario()
        return jsonify({"ok": True, "message": "Analysis session aborted successfully"})
    except Exception as e:
        logging.exception("Error aborting analysis session")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/get-all-input-datasets", methods=["GET"])
def get_all_input_datasets_api():
    """Return available dataset names for multi-select."""
    try:
        datasets = list_zone_datasets(flow_zone_name="Input Datasets")
        datasets = [{"label": d, "value": d} for d in datasets]
        return jsonify({"ok": True, "all_input_datasets": datasets})
    except Exception as e:
        logging.exception("Error getting all input datasets")
        return jsonify({"ok": False, "error": str(e)}), 500


@analysis_session_api.route("/get-dataset-columns", methods=["GET"])
def get_dataset_columns_api():
    """Get the columns of a dataset."""
    try:
        dataset_name = request.args.get("dataset_name")
        if not dataset_name:
            return jsonify({"ok": False, "error": "dataset_name is required"}), 400
        columns = get_dataset_columns(dataset_name)
        return jsonify({"ok": True, "columns": columns})
    except Exception as e:
        logging.exception("Error getting dataset columns")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/get-all-analysis-types", methods=["GET"])
def get_all_analysis_types_api():
    """
    Get all analysis types.
    """
    try:
        analytic_tools = get_available_analytic_tools()
        analytic_tools = [{'label': row['analytic_tool_name'], 'value': row['analytic_tool_id']} for _, row in analytic_tools.iterrows()]
        return jsonify({"ok": True, "analytic_tools": analytic_tools})
    except Exception as e:
        logging.exception("Error getting all analysis types")
        return jsonify({"ok": False, "error": str(e)}), 500


@analysis_session_api.route("/get-all-analysis-session-metadata", methods=["GET"])
def get_all_analysis_session_metadata_api():
    """
    Get all analysis session metadata.
    """
    try:
        all_sessions = get_all_analysis_session_metadata()
        all_sessions = convert_nan_to_none(all_sessions.to_dict(orient="records"))
        return jsonify({"ok": True, "all_sessions": all_sessions})
    except Exception as e:
        logging.exception("Error getting all analysis session metadata")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/get-analysis-report", methods=["GET"])
def get_analysis_report_api():
    """
    Get the analysis report for a given analysis session ID.
    Updates metadata with end_datetime and status when report is found.
    """
    try:
        analysis_session_id = request.args.get("analysis_session_id")
        #analysis_session_id = "report_3"
        if not analysis_session_id:
            return jsonify({"ok": False, "error": "analysis_session_id is required"}), 400
        
        report = get_analysis_report(analysis_session_id)
        if report:
            # Update metadata with end_datetime and status when report is available
            #from datetime import datetime
            #existing_metadata = get_analysis_session_metadata(analysis_session_id)
            #if existing_metadata:
            #    # Preserve existing metadata and update end_datetime and status
            #    existing_metadata["end_datetime"] = datetime.utcnow().isoformat() + "Z"
            #    existing_metadata["status"] = "completed"
            #    update_analysis_session_metadata(
            #        analysis_session_id=analysis_session_id,
            #        metadata=existing_metadata
            #    )
            
            return jsonify({"ok": True, "report": report})
        else:
            return jsonify({"ok": False, "error": "Report not found"}), 404
    except Exception as e:
        logging.exception("Error getting analysis report")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/delete-analysis-session", methods=["POST"])
def delete_analysis_session_api():
    """
    Delete an analysis session.
    """
    try:
        data = request.get_json()
        if not data:
            return jsonify({"ok": False, "error": "No data received"}), 400
        analysis_session_id = data.get("analysis_session_id")
        if not analysis_session_id:
            return jsonify({"ok": False, "error": "analysis_session_id is required"}), 400
        delete_analysis_session(analysis_session_id)
        return jsonify({"ok": True, "message": "Analysis session deleted successfully"})
    except Exception as e:
        logging.exception("Error deleting analysis session")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/get-all-report-templates", methods=["GET"])
def get_all_report_templates():
    """
    Get all report templates.
    """
    try:
        report_template_data = {
            'default_report_id': 'default_report_template.html',
            'choice': ['default_report_template.html']
        }
        return jsonify({"ok": True, "report_template_data": report_template_data})
    except Exception as e:
        logging.exception("Error getting all report templates")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/get-last-analysis-run-status", methods=["GET"])
def get_last_analysis_run_status_api():
    """
    Get the last analysis session scenario run status.
    Returns the most recent run's start time.
    """
    try:
        run_insights_df = get_main_analysis_session_scenario_run_status()
        
        if run_insights_df is not None and not run_insights_df.empty:
            # Get the first row (most recent run, already sorted by start_dt_utc descending)
            last_run = run_insights_df.iloc[0].to_dict()
            # Convert to dict and handle NaN values
            last_run_cleaned = convert_nan_to_none([last_run])[0]
            return jsonify({"ok": True, "last_run": last_run_cleaned})
        else:
            return jsonify({"ok": True, "last_run": None})
    except Exception as e:
        logging.exception("Error getting last analysis run status")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/update-analysis-session-params-in-flow", methods=["POST"])
def update_analysis_session_params_in_flow_api():
    """
    Update the analysis session params in the flow.
    """
    try:
        data = request.get_json()
        update_analysis_session_params_in_flow(analysis_session_id=data.get("analysis_session_id"))
        return jsonify({"ok": True, "message": "Analysis session params updated in flow successfully"})
    except Exception as e:
        logging.exception("Error updating analysis session params in flow")
        return jsonify({"ok": False, "error": str(e)}), 500

@analysis_session_api.route("/get-report-assistant-webapp-path", methods=["GET"])
def get_report_assistant_webapp_path_api():
    """
    Get the relative path to the report assistant webapp.
    """
    try:
        relative_path = get_report_assistant_webapp_relative_path()
        return jsonify({"ok": True, "relative_path": relative_path})
    except Exception as e:
        logging.exception("Error getting report assistant webapp path")
        return jsonify({"ok": False, "error": str(e)}), 500