import { defineStore } from 'pinia'
import { ref } from 'vue'
import { API } from '@/Api'

const STORAGE_KEY_CURRENT_SESSION = 'app_current_analysis_session'
const STORAGE_KEY_HISTORICAL_SESSIONS = 'app_historical_analysis_sessions'

/**
 * Load data from localStorage
 */


/**
 * Save data to localStorage
 */
function saveToStorage<T>(key: string, data: T): void {
  try {
    localStorage.setItem(key, JSON.stringify(data))
  } catch (error) {
    console.error(`Error saving to localStorage (${key}):`, error)
  }
}

export interface AnalysisSession {
  analysisSessionId: string
  status: 'pending' | 'running' | 'completed' | 'failed'
  reportHtml?: string
}

export interface HistoricalAnalysisSession {
  id: string
  inputDataset?: string
  analysisToPerform?: string[]
  inferTargetVariable?: boolean
  additionalContext?: string
  status?: 'completed' | 'running' | 'failed' | 'pending'
  [key: string]: any // Allow additional fields from API
}

export const useAnalysisStore = defineStore('analysis', () => {
  // Current analysis session
  const currentSession = ref<AnalysisSession | null>(null)
  // Historical analysis sessions
  const historicalSessions = ref<HistoricalAnalysisSession[]>([])


  /**
   * Set current analysis session
   */
  function setCurrentSession(session: AnalysisSession): void {
    currentSession.value = session
    saveToStorage(STORAGE_KEY_CURRENT_SESSION, session)
  }

  /**
   * Update session status
   */
  function updateSessionStatus(status: AnalysisSession['status']): void {
    if (currentSession.value) {
      currentSession.value.status = status
      saveToStorage(STORAGE_KEY_CURRENT_SESSION, currentSession.value)
    }
  }

  /**
   * Set report HTML for current session
   */
  function setReportHtml(reportHtml: string): void {
    if (currentSession.value) {
      currentSession.value.reportHtml = reportHtml
      currentSession.value.status = 'completed'
      saveToStorage(STORAGE_KEY_CURRENT_SESSION, currentSession.value)
    }
  }

  /**
   * Clear current session
   */
  function clearCurrentSession(): void {
    currentSession.value = null
    localStorage.removeItem(STORAGE_KEY_CURRENT_SESSION)
  }

  /**
   * Fetch analysis report from API
   */
  async function fetchAnalysisReport(analysisSessionId: string): Promise<string | null> {
    try {
      const response = await API.getAnalysisReport(analysisSessionId)
      if (response.data.ok && response.data.report) {
        return response.data.report
      }
      return null
    } catch (error) {
      console.error('Error fetching analysis report:', error)
      return null
    }
  }

  // Flag to stop polling
  let shouldStopPolling = false

  /**
   * Stop polling for report
   */
  function stopPolling(): void {
    shouldStopPolling = true
  }

  /**
   * Poll for report until available or timeout
   */
  async function pollForReport(
    analysisSessionId: string,
    maxAttempts: number = 5,
    intervalMs: number = 2000
  ): Promise<string | null> {
    // Stop any existing polling first
    shouldStopPolling = true
    
    // Wait a bit to ensure previous polling stops
    await new Promise(resolve => setTimeout(resolve, 100))
    
    // Reset flag for new polling
    shouldStopPolling = false
    
    for (let attempt = 0; attempt < maxAttempts; attempt++) {
      // Check if polling should stop
      if (shouldStopPolling) {
        console.log('Polling stopped by user')
        return null
      }

      const report = await fetchAnalysisReport(analysisSessionId)
      if (report) {
        setReportHtml(report)
        return report
      }
      
      // Wait before next attempt
      await new Promise(resolve => setTimeout(resolve, intervalMs))
    }
    
    console.log(`Polling completed after ${maxAttempts} attempts without finding report`)
    // Update status to failed if polling completed without finding report
    if (currentSession.value && currentSession.value.status === 'running') {
      updateSessionStatus('failed')
    }
    return null
  }

  
  /**
   * Fetch all historical sessions from API
   */
  async function fetchAllHistoricalSessions(): Promise<void> {
    try {
      const response = await API.getAnalysisSessions()
      if (response.data.ok && response.data.all_sessions) {
        historicalSessions.value = response.data.all_sessions
        saveToStorage(STORAGE_KEY_HISTORICAL_SESSIONS, historicalSessions.value)
      } else {
        console.error('Error loading historical sessions:', response.data.error)
      }
    } catch (error) {
      console.error('Error fetching historical sessions:', error)
    }
  }

  /**
   * Refresh historical sessions (force API call)
   */
  async function refreshHistoricalSessions(): Promise<void> {
    await fetchAllHistoricalSessions()
  }

  // Polling interval for historical sessions
  let historicalSessionsPollingInterval: ReturnType<typeof setInterval> | null = null

  /**
   * Start polling for historical sessions updates
   */
  function startHistoricalSessionsPolling(intervalMs: number = 10000): void {
    // Clear any existing interval
    stopHistoricalSessionsPolling()
    
    // Fetch immediately
    fetchAllHistoricalSessions()
    
    // Then poll at interval
    historicalSessionsPollingInterval = setInterval(() => {
      fetchAllHistoricalSessions()
    }, intervalMs)
  }

  /**
   * Stop polling for historical sessions updates
   */
  function stopHistoricalSessionsPolling(): void {
    if (historicalSessionsPollingInterval) {
      clearInterval(historicalSessionsPollingInterval)
      historicalSessionsPollingInterval = null
    }
  }

  // Don't load current session from storage on init - it will be cleared on page refresh
  // This ensures no stale reports are shown after refresh
  // Don't load historical sessions from storage on init - they will be fetched from API
  // This ensures fresh data on page refresh

  return {
    // State
    currentSession,
    historicalSessions,
    // Actions
    setCurrentSession,
    updateSessionStatus,
    setReportHtml,
    clearCurrentSession,
    fetchAnalysisReport,
    pollForReport,
    stopPolling,
    fetchAllHistoricalSessions,
    refreshHistoricalSessions,
    startHistoricalSessionsPolling,
    stopHistoricalSessionsPolling,
  }
})

