import { defineStore } from 'pinia'
import { ref } from 'vue'
import { API } from '@/Api'
import type { AnalysisOption, SelectOption } from '@/utils/analysisTypes'

const STORAGE_KEY_ANALYSIS_TYPES = 'app_analysis_types'
const STORAGE_KEY_INPUT_DATASETS = 'app_input_datasets'

/**
 * Load data from localStorage
 */
function loadFromStorage<T>(key: string): T | null {
  try {
    const item = localStorage.getItem(key)
    return item ? JSON.parse(item) : null
  } catch (error) {
    console.error(`Error loading from localStorage (${key}):`, error)
    return null
  }
}

/**
 * Save data to localStorage
 */
function saveToStorage<T>(key: string, data: T): void {
  try {
    localStorage.setItem(key, JSON.stringify(data))
  } catch (error) {
    console.error(`Error saving to localStorage (${key}):`, error)
  }
}

export const useOptionsStore = defineStore('options', () => {
  // Analysis types
  const analysisTypes = ref<AnalysisOption[]>([])
  const analysisTypesLoaded = ref(false)

  // Input datasets
  const inputDatasets = ref<SelectOption[]>([])
  const inputDatasetsLoaded = ref(false)

  /**
   * Load analysis types from storage or API
   */
  async function loadAnalysisTypes(forceRefresh: boolean = false): Promise<void> {
    // If forcing refresh, clear localStorage and reset the loaded flag to ensure fresh data
    if (forceRefresh) {
      localStorage.removeItem(STORAGE_KEY_ANALYSIS_TYPES)
      analysisTypesLoaded.value = false
      // Don't clear current values immediately - wait for new data to arrive
    }
    
    // If already loaded and not forcing refresh, return
    if (analysisTypesLoaded.value && !forceRefresh) {
      return
    }

    // Try to load from localStorage first (unless forcing refresh)
    if (!forceRefresh) {
      const stored = loadFromStorage<AnalysisOption[]>(STORAGE_KEY_ANALYSIS_TYPES)
      if (stored && stored.length > 0) {
        analysisTypes.value = stored
        analysisTypesLoaded.value = true
        return
      }
    }

    // Load from API
    try {
      const response = await API.getAllAnalysisTypes()
      console.log('API response for analysis types:', response.data)
      if (response.data.ok) {
        // API returns 'analytic_tools' but we expect 'all_analysis_types' or 'analytic_tools'
        const data = response.data.all_analysis_types || response.data.analytic_tools
        console.log('Extracted analysis types data:', data)
        if (data && Array.isArray(data) && data.length > 0) {
          analysisTypes.value = data as AnalysisOption[]
          analysisTypesLoaded.value = true
          // Save to localStorage
          saveToStorage(STORAGE_KEY_ANALYSIS_TYPES, analysisTypes.value)
          console.log('Analysis types loaded successfully:', analysisTypes.value.length, 'items')
        } else {
          console.warn('No analysis types returned from API or empty array. Data:', data)
          // Only clear if we got an empty response
          if (forceRefresh) {
            analysisTypes.value = []
          }
        }
      } else {
        console.error('API returned ok=false:', response.data)
      }
    } catch (error) {
      console.error('Error loading analysis types:', error)
      // On error, don't clear existing data unless it was a forced refresh with no fallback
    }
  }

  /**
   * Load input datasets from storage or API
   */
  async function loadInputDatasets(forceRefresh: boolean = false): Promise<void> {
    // If forcing refresh, clear localStorage and reset the loaded flag to ensure fresh data
    if (forceRefresh) {
      localStorage.removeItem(STORAGE_KEY_INPUT_DATASETS)
      inputDatasetsLoaded.value = false
      // Don't clear current values immediately - wait for new data to arrive
    }
    
    // If already loaded and not forcing refresh, return
    if (inputDatasetsLoaded.value && !forceRefresh) {
      return
    }

    // Try to load from localStorage first (unless forcing refresh)
    if (!forceRefresh) {
      const stored = loadFromStorage<SelectOption[]>(STORAGE_KEY_INPUT_DATASETS)
      if (stored && stored.length > 0) {
        inputDatasets.value = stored
        inputDatasetsLoaded.value = true
        return
      }
    }

    // Load from API
    try {
      const response = await API.getAllInputDatasets()
      if (response.data.ok) {
        // API returns 'all_input_datasets'
        const data = response.data.all_input_datasets
        if (data && Array.isArray(data) && data.length > 0) {
          inputDatasets.value = data as SelectOption[]
          inputDatasetsLoaded.value = true
          // Save to localStorage
          saveToStorage(STORAGE_KEY_INPUT_DATASETS, inputDatasets.value)
        } else {
          console.warn('No input datasets returned from API')
          // Only clear if we got an empty response
          if (forceRefresh) {
            inputDatasets.value = []
          }
        }
      }
    } catch (error) {
      console.error('Error loading input datasets:', error)
    }
  }

  /**
   * Clear all stored data (useful for testing or reset)
   */
  function clearStorage(): void {
    localStorage.removeItem(STORAGE_KEY_ANALYSIS_TYPES)
    localStorage.removeItem(STORAGE_KEY_INPUT_DATASETS)
    analysisTypes.value = []
    inputDatasets.value = []
    analysisTypesLoaded.value = false
    inputDatasetsLoaded.value = false
  }

  return {
    // State
    analysisTypes,
    analysisTypesLoaded,
    inputDatasets,
    inputDatasetsLoaded,
    // Actions
    loadAnalysisTypes,
    loadInputDatasets,
    clearStorage,
  }
})

