import re
import datetime
import pytz


## Pattern to detect a dataiku ISO_8601 date in UTC:
DATAIKU_ISO_8601_UTC_DATE_PATTERN = ".+Z$"

## Format of a Dataiku ISO_8601 date:
DATAIKU_ISO_8601_DATE_FORMAT_IN_UTC = (
    "%Y-%m-%dT%H:%M:%S.%fZ"  # # Example: '2013-05-30T15:16:13.764Z'
)

## Format of a Dataiku ISO_8601 date that has a timezone:
DATAIKU_ISO_8601_DATE_FORMAT_WITH_TIME_ZONE = (
    "%Y-%m-%dT%H:%M:%S.%f%z"  # Example: '2013-05-30T15:16:13.764+0200'
)


def list_pytz_timezones():
    """
    Lists all the timezones that we can interact with in pytz.

    :returns: pytz_timezones: list: List of all pytz timezones.
    """
    pytz_timezones = pytz.all_timezones
    return pytz_timezones


def check_timezone_is_defined_in_pytz(focus_timezone: str):
    """
    Checks whether a time zone is defined or not in pytz.

    :param: focus_timezone: str: Timezone to check.
    """
    pytz_timezones = list_pytz_timezones()
    if focus_timezone not in pytz_timezones:
        log_message = (
            f"You asked for using the time zone '{focus_timezone}' that is not defined in pytz. "
            f"Please choose a time zone in '{pytz_timezones}'"
        )
        raise ValueError(log_message)
    pass


# Timezones dates conversions:
def convert_datetime_date_in_timezone(
    datetime_date: datetime.datetime, target_timezone: str = "UTC"
):
    """
    Converts a datetime date in another timezone.

    :param: datetime_date: datetime.datetime: The datetime date to convert in another timezone.
    :param: target_timezone: str: Timezone where to convert the date.
        Supported timezones are the ones listed in pytz using 'pytz.all_timezones'.
    :returns: datetime_date_in_timezone: datetime.datetime: 'datetime_date' converted in 'target_timezone'.
    """
    check_timezone_is_defined_in_pytz(target_timezone)
    datetime_date_in_timezone = datetime_date.astimezone(pytz.timezone(target_timezone))
    return datetime_date_in_timezone


def from_dss_string_date_to_datetime(dss_string_date, target_timezone="UTC"):
    """
    Converts a dataiku DSS string datetime value into a datetime object.

    :param: dss_string_date: str: The string date to convert. This date might be formated with
        dataiku non-ambiguous dates format, also known as 'ISO-8601'.
    :param: target_timezone: str: Timezone where to convert the date.
        Supported timezones are the ones listed in pytz using 'pytz.all_timezones'.

    :returns: datetime_date: datetime.datetime: datetime date associated with the 'dss_string_date'.
    """
    if re.match(DATAIKU_ISO_8601_UTC_DATE_PATTERN, dss_string_date):
        date_format = DATAIKU_ISO_8601_DATE_FORMAT_IN_UTC
    else:
        date_format = DATAIKU_ISO_8601_DATE_FORMAT_WITH_TIME_ZONE

    datetime_date = datetime.datetime.strptime(dss_string_date, date_format)
    datetime_date = convert_datetime_date_in_timezone(datetime_date, target_timezone)
    return datetime_date


def from_datetime_to_dss_string_date(datetime_date, target_timezone="UTC"):
    """
    Converts a datetime object into a dataiku DSS string datetime value.

    :param: datetime_date:  datetime.datetime: The datetime date to convert.
    :param: target_timezone: str: Timezone where to convert the date.
        Supported timezones are the ones listed in pytz using 'pytz.all_timezones'.

    :returns: dss_string_date: str: DSS string date associated with the 'datetime_date'.
    """
    datetime_date = convert_datetime_date_in_timezone(datetime_date, target_timezone)
    datetime_information = datetime_date.strftime("%Y-%m-%dT%H:%M:%S.%f")[
        :-3
    ]  # We remove the end of the string to get 'milliseconds' resolution instead of 'microseconds'.
    timezone_information = datetime_date.strftime("%z")
    dss_string_date = datetime_information + timezone_information
    return dss_string_date
