import { defineStore } from "pinia";
import { ref, computed, watch } from "vue";
import axios from "@/api/index";
import { MultiSelectUI, MultiSelectType, similarStudiesFilteringKeys, candidateSitesFilteringKeys, GroupableCandidateSites } from "@/models";
import { CandidateSite, NovelStudy, SimilarStudy, } from "@/models/backend";
import { UIOptions, StudySummary } from "@/models/backend";
import { cloneDeep, isEqual, snakeCase } from "lodash";
import { Loading } from "quasar";
import { studyStatusColorMapping, defaultBadgeColor, studyEnrollmentRateColorMapping } from "@/constants";
import { useI18n } from "vue-i18n";
import { localizedFormat } from "@/utils/localized-format";
import { GroupableSites } from "@/models";
import { extractLatLongFromGeoPoint, extractLatLongFromGeoPoints, getGeoJson } from "@/utils/geo";


export enum StudyType {
    NOVEL = "Novel study",
    EXISTING = "Existing study"
}

export enum StudySummaryTabContentType {
    NOVEL,
    EXISTING,
    EMPTY,
}

export const useStudiesStore = defineStore("studies", () => {

    const openThirdTab = ref(false);

    const studyTypes = ref<StudyType[]>([StudyType.EXISTING, StudyType.NOVEL]);

    const selectedStudyType = ref<StudyType | null>(null);

    const uiOptions = ref<UIOptions | null>(null);

    const selectedStudy = ref<string | null>(null);

    const selectedStudySummary = ref<StudySummary | null>(null);

    const loadingStudySummary = ref(false);

    const novelStudy = ref<NovelStudy | null>(null);



    const loadingSimilarStudies = ref(false);

    // Similar studies
    const selectedStudySimilarStudies = ref<SimilarStudy[] | null>(null);

    const novelStudySimilarStudies = ref<SimilarStudy[] | null>(null);

    const similarStudiesFiltersUIExisting = ref<MultiSelectUI[] | null>(null);

    const similarStudiesFiltersUINovel = ref<MultiSelectUI[] | null>(null);

    const similarStudiesFiltersExisting = ref<{ [key: string]: string[] } | null>(null);

    const similarStudiesFiltersNovel = ref<{ [key: string]: string[] } | null>(null);

    // Candidate sites
    const selectedCandidateSites = ref<CandidateSite[] | null>(null);

    const novelCandidateSites = ref<CandidateSite[] | null>(null);

    const candidateSitesFilters = ref<{ [key: string]: string[] } | null>(null);


    // Make studies & sites tab global
    const selectedStudiesSitesTab = ref<string>("similar_studies")




    const { t } = useI18n();

    const novelStudyEligibilityTexts = computed(() => {
        return novelStudy.value == null ? null : {
            cohortSex: novelStudy.value.cohortSex != null ? `SEX: ${novelStudy.value.cohortSex}` : null,
            cohortAge: novelStudy.value.cohortAge.length > 0 ? `AGES: ${novelStudy.value.cohortAge.join(" - ")}` : null,
            healthyVolunteers: `Health Volunteers: ${novelStudy.value.healthyVolunteers ? "True" : "False"}`
        }
    })

    const hydrated = ref(false);

    const studySummaryTabContentType = computed(() => {
        if (selectedStudyType.value == StudyType.NOVEL && novelStudy.value != null) {
            return StudySummaryTabContentType.NOVEL;
        } else if (
            selectedStudyType.value == StudyType.EXISTING &&
            selectedStudy.value != null
        ) {
            return StudySummaryTabContentType.EXISTING;
        } else {
            return StudySummaryTabContentType.EMPTY;
        }
    });

    const headerTitle = computed(() => {
        if (selectedStudyType.value == StudyType.EXISTING && selectedStudy.value != null) {
            return {
                title: `ClinicalTrials.gov.ID ${selectedStudy.value}`,
                href: `https://clinicaltrials.gov/study/${selectedStudy.value}`
            }
        }
        return null;
    })

    const selectedStudyStatusData = computed(() => {
        if (selectedStudySummary.value != null && selectedStudySummary.value.design != null) {
            const status = selectedStudySummary.value.design.OverallStatus ? selectedStudySummary.value.design.OverallStatus : 'NA';
            const colors = studyStatusColorMapping[status] ? studyStatusColorMapping[status] : defaultBadgeColor;
            return {
                text: status,
                ...colors
            }
        }
        return {
            text: 'NA',
            ...defaultBadgeColor
        }
    })

    const selectedStudyEnrollementRate = computed(() => {
        if (selectedStudySummary.value != null && selectedStudySummary.value.score != null) {
            const score = selectedStudySummary.value.score.Enrolment_rate_combined ? selectedStudySummary.value.score.Enrolment_rate_combined : 'NA';
            const colors = studyEnrollmentRateColorMapping[score] ? studyEnrollmentRateColorMapping[score] : defaultBadgeColor;
            return {
                text: score,
                ...colors
            }
        }
        return {
            text: 'NA',
            ...defaultBadgeColor
        }
    })

    const selectedStudyEnrollementText = computed(() => {
        let text = ''
        if (selectedStudySummary.value != null && selectedStudySummary.value.design != null) {
            if (selectedStudySummary.value.design.EnrollmentCount != null) {
                text += String(selectedStudySummary.value.design.EnrollmentCount)
                if (selectedStudySummary.value.design.EnrollmentType != null) {
                    text += ` ${selectedStudySummary.value.design.EnrollmentType}`
                }
                return text
            }
            return 'NA'
        }
        return 'NA'
    })

    const selectedStudyPatientEligibilityTexts = computed(() => {
        if (selectedStudySummary.value != null && selectedStudySummary.value.eligibility != null) {
            return {
                sex: `SEX: ${selectedStudySummary.value.eligibility.Sex ? selectedStudySummary.value.eligibility.Sex : 'NA'}`,
                ages: `AGES: ${selectedStudySummary.value.eligibility.MinimumAge ? selectedStudySummary.value.eligibility.MinimumAge : 'NA'} - ${selectedStudySummary.value.eligibility.MaximumAge ? selectedStudySummary.value.eligibility.MaximumAge : 'NA'}`,
                healthVolunteers: `Health Volunteers: ${selectedStudySummary.value.eligibility.HealthyVolunteers != null ? String(selectedStudySummary.value.eligibility.HealthyVolunteers) : 'NA'}`
            }
        }
        return {
            sex: 'SEX: NA',
            ages: 'AGES: NA - NA',
            healthVolunteers: 'Health Volunteers: NA'
        }
    })

    const selectedStudySummaryCardsResults = computed(() => {
        let timeline: Record<string, string | string[]> = {
            'start_date': 'NA',
            'primary_completion_date': 'NA',
            'completion_date': 'NA'
        }
        let sponsor: Record<string, string | string[]> = {
            'lead_sponsor': 'NA',
            'collaborators': 'NA',
            'lead_sponsor_class': 'NA'
        }
        let study: Record<string, string | string[]> = {
            'type': 'NA',
            'phase': 'NA'
        }
        if (selectedStudySummary.value != null) {
            if (selectedStudySummary.value.design != null) {
                timeline = {
                    ...timeline,
                    'start_date': selectedStudySummary.value.design.StartDate_parsed ? localizedFormat(selectedStudySummary.value.design.StartDate_parsed, t('date-fns_date_short')) : 'NA',
                    'completion_date': selectedStudySummary.value.design.CompletionDate_parsed ? localizedFormat(selectedStudySummary.value.design.CompletionDate_parsed, t('date-fns_date_short')) : 'NA',
                    'primary_completion_date': selectedStudySummary.value.design.PrimaryCompletionDate_parsed ? localizedFormat(selectedStudySummary.value.design.PrimaryCompletionDate_parsed, t('date-fns_date_short')) : 'NA',
                }

                if (selectedStudySummary.value.info?.LastUpdatePostDate_parsed != null) {
                    timeline = {
                        ...timeline,
                        'last_update_posted': localizedFormat(selectedStudySummary.value.info?.LastUpdatePostDate_parsed, t('date-fns_date_short'))
                    }
                }

                study = {
                    ...study,
                    'type': selectedStudySummary.value.design.StudyType ? selectedStudySummary.value.design.StudyType : 'NA',
                    'phase': selectedStudySummary.value.design.Phase ? selectedStudySummary.value.design.Phase : 'NA'
                }
                sponsor = {
                    ...sponsor,
                    'lead_sponsor': selectedStudySummary.value.design.LeadSponsorName ? selectedStudySummary.value.design.LeadSponsorName : 'NA',
                    'lead_sponsor_class': selectedStudySummary.value.design.LeadSponsorClass ? selectedStudySummary.value.design.LeadSponsorClass : 'NA'
                }
            }
            if (selectedStudySummary.value.collaborators != null) {
                if (selectedStudySummary.value.collaborators.length > 0) {
                    sponsor = {
                        ...sponsor,
                        'collaborators': [...selectedStudySummary.value.collaborators].map((value) => value.Collaborator_name),

                    }
                }
            }
            if (selectedStudySummary.value.investigators != null) {
                if (selectedStudySummary.value.investigators.length > 0) {
                    sponsor = {
                        ...sponsor,
                        'investigators': [...selectedStudySummary.value.investigators].map((value) => value.Investigators)
                    }
                }
            }
        }
        return {
            timeline,
            sponsor,
            study
        }
    })

    const selectedStudyArmsData = computed(() => {
        const armsData: {
            title: string;
            chipText: string;
            description: string;
            interventions: Record<string, string[]>
        }[] = []
        if (selectedStudySummary.value != null && selectedStudySummary.value.arms != null) {
            for (const arm of selectedStudySummary.value.arms) {
                const interventionNames: string[] = arm.ArmGroup_interventionNames ? JSON.parse(arm.ArmGroup_interventionNames) : [];
                const interventions: Record<string, string[]> = {};
                for (const armIntervention of interventionNames) {
                    const splittedVals = armIntervention.split(':');
                    if (splittedVals.length > 1) {
                        const key = splittedVals[0].trim()
                        const val = splittedVals[1].trim()
                        if (interventions[key] != null) {
                            interventions[key].push(val);
                        } else {
                            interventions[key] = [val];
                        }

                    }

                }
                const armData = {
                    title: arm.ArmGroup_label ? arm.ArmGroup_label : 'NA',
                    chipText: arm.ArmGroup_type ? arm.ArmGroup_type : 'NA',
                    description: arm.ArmGroup_description ? arm.ArmGroup_description : 'NA',
                    interventions: interventions
                }
                armsData.push(armData);
            }
        }
        return armsData
    })

    const selectedStudyGroupableSites = computed(() => {
        const data: GroupableSites = {};
        if (selectedStudySummary.value != null && selectedStudySummary.value.sites != null) {
            for (const site of selectedStudySummary.value.sites) {
                const countryLabel = site.Site_Country != null ? site.Site_Country : 'NA'
                const countryKey = snakeCase(countryLabel);
                const regionLabel = site.Site_Region != null ? site.Site_Region : (site.Site_City != null ? site.Site_City : 'NA')
                const regionKey = snakeCase(regionLabel)
                if (data[countryKey] == null) {
                    data[countryKey] = { label: countryLabel, children: {} }
                }
                if (data[countryKey].children[regionKey] == null) {
                    data[countryKey].children[regionKey] = {
                        label: regionLabel,
                        data: []
                    }
                }

                data[countryKey].children[regionKey].data.push({
                    Facility_preferred_name: site.Facility_preferred_name ? site.Facility_preferred_name : 'NA',
                    Facility_ID: site.Facility_ID ? site.Facility_ID : 'NA',
                    Site_status: site.Site_status ? site.Site_status : 'NA',
                    Location: site.Location ? site.Location : 'NA'
                })
            }
        }
        return data;
    });

    const geoJsonSitesData = computed(() => {
        if (selectedStudySummary.value != null && selectedStudySummary.value.sites != null) {
            return selectedStudySummary.value.sites.map((value) => {
                return {
                    ...value,
                    latlong: extractLatLongFromGeoPoint(value.GeoZip)
                }
            }).filter((value) => value.latlong != null);
        }
        return null;
    });



    async function setSelectedStudy(studyId: string | null) {
        if (studyId !== selectedStudy.value) {
            selectedStudy.value = studyId;
            if (studyId != null) {
                Loading.show();
                await getStudySummary(studyId);
                await getSimilarStudiesExisting(studyId);
                Loading.hide();
            } else {
                selectedStudySummary.value = null;
            }
        }
    }

    async function setNovelStudy(study: NovelStudy) {
        novelStudy.value = cloneDeep({ ...study });
        Loading.show();
        await getSimilarStudiesNovel(study);
        Loading.hide();
    }

    async function getStudiesIds() {
        try {
            const response = await axios.get<UIOptions>(`/api/studies/ui_options`);
            uiOptions.value = response.data
        } catch (e) {
            // DEBUG
        }
    }

    async function getStudySummary(id: string) {
        selectedStudySummary.value = null;
        loadingStudySummary.value = true;
        try {
            const response = await axios.get<StudySummary>(`/api/studies/summary/${id}`);
            selectedStudySummary.value = response.data;
        } catch (e) {
            // DEBUG
        } finally {
            loadingStudySummary.value = false
        }
    }

    async function getSimilarStudiesExisting(id: string) {
        selectedStudySimilarStudies.value = null;
        similarStudiesFiltersExisting.value = null;
        similarStudiesFiltersUIExisting.value = null;

        selectedCandidateSites.value = null;

        loadingSimilarStudies.value = true;
        try {
            const response = await axios.get<{ studies: SimilarStudy[], sites: CandidateSite[] }>(`/api/study/similar/${id}`);
            selectedStudySimilarStudies.value = response.data.studies;
            selectedCandidateSites.value = response.data.sites;
            const { ui, initialFilters } = computeSimilarStudiesFilterOptions(response.data.studies);
            similarStudiesFiltersUIExisting.value = ui;
            similarStudiesFiltersExisting.value = initialFilters;

        } catch (e) {
            // DEBUG
        } finally {
            loadingSimilarStudies.value = false;
        }
    }

    async function getSimilarStudiesNovel(novelStudy: NovelStudy) {
        novelStudySimilarStudies.value = null;
        similarStudiesFiltersUINovel.value = null;
        similarStudiesFiltersNovel.value = null;

        novelCandidateSites.value = null;

        loadingSimilarStudies.value = true;

        try {
            const response = await axios.post<{ studies: SimilarStudy[], sites: CandidateSite[] }>(`/api/study/novel/similar`, novelStudy);
            novelStudySimilarStudies.value = response.data.studies;
            novelCandidateSites.value = response.data.sites;
            const { ui, initialFilters } = computeSimilarStudiesFilterOptions(response.data.studies);
            similarStudiesFiltersUINovel.value = ui;
            similarStudiesFiltersNovel.value = initialFilters;

        } catch (e) {
            // DEBUG
        } finally {
            loadingSimilarStudies.value = false;
        }
    }

    async function hydrate() {
        await getStudiesIds();
        hydrated.value = true;
    }

    function computeCandidateSitesFilterOptions(candidateSites: CandidateSite[]): { ui: MultiSelectUI[], initialFilters: { [key: string]: string[] } } {
        const results: MultiSelectUI[] = [];
        const initialFilters: { [key: string]: (string | number)[] } = {}
        for (const filteringData of candidateSitesFilteringKeys) {
            const result: MultiSelectUI = {
                ...filteringData,
                options: []
            }
            for (const site of candidateSites) {
                if (site[filteringData.key] != null) {
                    result.options.push(String(site[filteringData.key]))
                }
            }
            result.options = [... new Set(result.options)];
            initialFilters[filteringData.key] = [...result.options];
            results.push(result);
        }

        return { ui: results, initialFilters: initialFilters as { [key: string]: string[] } }
    }

    function computeSimilarStudiesFilterOptions(similarStudies: SimilarStudy[]): { ui: MultiSelectUI[], initialFilters: { [key: string]: string[] } } {
        const results: MultiSelectUI[] = [];
        const initialFilters: { [key: string]: (string | number)[] } = {}
        for (const filteringData of similarStudiesFilteringKeys) {
            const result: MultiSelectUI = {
                ...filteringData,
                options: []
            }
            for (const study of similarStudies) {
                if (study[filteringData.key] != null) {
                    if (!(filteringData.key == 'MeshTerm_Conditions')) {
                        result.options.push(study[filteringData.key])
                    } else {
                        const localOptions = (study[filteringData.key] ?? '').split(".");
                        result.options = result.options.concat(localOptions);
                    }

                }
            }
            result.options = [...new Set(result.options)]
            initialFilters[filteringData.key] = [...result.options];
            results.push(result);
        }
        return { ui: results, initialFilters: initialFilters as { [key: string]: string[] } };
    }

    const similarStudiesFiltersUI = computed(() => {
        if (selectedStudyType.value == StudyType.EXISTING) {
            return similarStudiesFiltersUIExisting.value;
        } else {
            return similarStudiesFiltersUINovel.value;
        }
    });

    const similarStudiesFilters = computed(() => {
        if (selectedStudyType.value == StudyType.EXISTING) {
            return similarStudiesFiltersExisting.value;
        } else {
            return similarStudiesFiltersNovel.value;
        }
    });


    const filteredSimilarStudies = computed(() => {
        if (selectedStudyType.value == StudyType.EXISTING) {
            if (selectedStudySimilarStudies.value != null && similarStudiesFiltersExisting.value != null) {
                const { initialFilters } = computeSimilarStudiesFilterOptions(selectedStudySimilarStudies.value);
                return filterSimilarStudies(selectedStudySimilarStudies.value, similarStudiesFiltersExisting.value, initialFilters);
            }
        } else {
            if (novelStudySimilarStudies.value != null && similarStudiesFiltersNovel.value != null) {
                const { initialFilters } = computeSimilarStudiesFilterOptions(novelStudySimilarStudies.value);
                return filterSimilarStudies(novelStudySimilarStudies.value, similarStudiesFiltersNovel.value, initialFilters);
            }
        }
        return [];
    });

    const similarStudiesCandidateSites = computed(() => {
        const filteredStudiesIds = filteredSimilarStudies.value.map((study) => study.Study_id);
        const sites = selectedStudyType.value == StudyType.EXISTING ? selectedCandidateSites.value : novelCandidateSites.value;
        if (sites != null && sites.length > 0) {
            const filteredSites = sites.filter((site) => {
                const siteStudyIds: string[] = site.Study_info.map((value) => value.length > 0 ? value[0] : null).filter((value) => value != null) as string[];
                return siteStudyIds.some((value) => filteredStudiesIds.includes(value));
            });
            return filteredSites.map((site) => {
                const filterIndexes = site.Study_info.map((studyInfo, index) => {
                    if (studyInfo.length > 0 && studyInfo[0] != null && filteredStudiesIds.includes(studyInfo[0])) {
                        return index
                    }
                }).filter((val) => val != null) as number[];

                return {
                    ...site,
                    Study_info: filterIndexes.map((val) => site.Study_info[val]),
                    Similarity: filterIndexes.map((val) => site.Similarity[val])
                }
            })
        }
        return [];
    });

    const finalCandidateSites = computed(() => {
        const { initialFilters } = computeCandidateSitesFilterOptions(similarStudiesCandidateSites.value);
        if (candidateSitesFilters.value) return filterCandidateSites(similarStudiesCandidateSites.value, candidateSitesFilters.value, initialFilters);
        else return similarStudiesCandidateSites.value;
    })

    const groupableCandidateSites = computed(() => {
        const data: GroupableCandidateSites = {};
        for (const site of finalCandidateSites.value) {
            const countryLabel = site.Country != null ? site.Country : 'NA'
            const countryKey = snakeCase(countryLabel);
            const regionLabel = site.Region_State != null ? site.Region_State : (site.Region_State != null ? site.City : 'NA')
            const regionKey = snakeCase(regionLabel)
            if (data[countryKey] == null) {
                data[countryKey] = { label: countryLabel, children: {} }
            }
            if (data[countryKey].children[regionKey] == null) {
                data[countryKey].children[regionKey] = {
                    label: regionLabel,
                    data: []
                }
            }
            data[countryKey].children[regionKey].data.push({
                Facility_preferred_name: site.Facility_preferred_name ? site.Facility_preferred_name : 'NA',
                Facility_ID: site.Facility_ID ? site.Facility_ID : 'NA',
                Has_competing_studies: site.Has_competing_studies,
                Country: site.Country != null ? site.Country : 'NA',
                Region_State: site.Region_State != null ? site.Region_State : 'NA',
                City: site.City != null ? site.City : 'NA',
                Zip: site.Zip != null ? site.Zip : 'NA',
            })
        }
        return data;
    })

    const candidateSitesTableData = computed(() => {

        return {
            title: 'tabs.studies_sites.candidate_sites_table_title',
            columns: [
                { name: 'Has_competing_studies', field: 'Has_competing_studies', label: 'Has Competing Studies', align: 'center' },
                { name: 'Facility_ID', field: 'Facility_ID', label: 'Site ID', align: 'center' },
                { name: 'Facility_preferred_name', field: 'Facility_preferred_name', label: 'Site Preferred Name', align: 'center' },
                { name: 'Similarity', field: 'Similarity', label: 'Similarity Score', sort: (a: string, b: string) => { return parseFloat(a) - parseFloat(b) }, sortable: true, align: 'center' },
                { name: 'Country', field: 'Country', label: 'Country', sortable: true, align: 'center' },
                { name: 'Region_State', field: 'Region_State', label: 'Region or State', sortable: true, align: 'center' },
                { name: 'City', field: 'City', label: 'City', sortable: true, align: 'center' },
                { name: 'Study_info', field: 'Study_info', label: 'Study info', align: 'center' }
            ],
            rows: finalCandidateSites.value.map((site) => {
                return {
                    ...site,
                    Similarity: (Math.round(site.Similarity.reduce((a, b) => a + b, 0) * 100) / 100).toFixed(2),
                    Study_info: JSON.stringify(site.Study_info.map((value) => {
                        return {
                            Study_id: value.length > 0 ? value[0] : 'NA',
                            Study_status: value.length > 1 ? value[1] : 'NA',
                            Site_status: value.length > 2 ? value[2] : 'NA'
                        }
                    }))
                }
            })
        }
    })

    watch(() => similarStudiesCandidateSites.value, (newVal, oldVal) => {
        if (!isEqual(oldVal, newVal)) {
            const { initialFilters } = computeCandidateSitesFilterOptions(similarStudiesCandidateSites.value);
            candidateSitesFilters.value = initialFilters;
        }
    }, { immediate: true, deep: true });

    const candidateSiteFiltersUI = computed(() => {
        const { ui } = computeCandidateSitesFilterOptions(similarStudiesCandidateSites.value);
        return ui;
    });

    function filterCandidateSites(sites: CandidateSite[], filters: { [key: string]: string[] }, initialFilters: { [key: string]: string[] }) {
        const result: CandidateSite[] = [];
        if (isEqual(initialFilters, filters)) return sites;
        for (const site of sites) {
            let toAdd = true;
            for (const key in filters) {
                const values = filters[key];
                const value = String(site[key]);
                if (value == null) toAdd = false;
                else {
                    if (!values.includes(value)) {
                        toAdd = false;
                    }
                }
            }
            if (toAdd) result.push(site);
        }
        return result;
    }


    function filterSimilarStudies(studies: SimilarStudy[], filters: { [key: string]: string[] }, initialFilters: { [key: string]: string[] }) {
        const result: SimilarStudy[] = [];
        if (isEqual(initialFilters, filters)) return studies;
        for (const study of studies) {
            let toAdd = true;
            for (const key in filters) {
                const values = filters[key];
                const value = study[key];
                if (value == null) toAdd = false;
                else {
                    if (key === 'MeshTerm_Conditions') {
                        if (!values.some((element) => value.includes(element))) {
                            toAdd = false;
                        }
                    } else {
                        if (!values.includes(value)) {
                            toAdd = false;
                        }
                    }
                }
            }
            if (toAdd) result.push(study);

        }
        return result;
    }

    return {
        studyTypes,
        selectedStudyType,
        hydrate,
        uiOptions,
        hydrated,
        selectedStudy,
        setSelectedStudy,
        novelStudy,
        setNovelStudy,
        studySummaryTabContentType,
        novelStudyEligibilityTexts,
        headerTitle,
        selectedStudySummary,
        loadingStudySummary,
        selectedStudyStatusData,
        selectedStudyEnrollementText,
        selectedStudyEnrollementRate,
        selectedStudyPatientEligibilityTexts,
        selectedStudySummaryCardsResults,
        selectedStudyArmsData,
        selectedStudyGroupableSites,
        loadingSimilarStudies,
        similarStudiesFiltersUIExisting,
        similarStudiesFiltersExisting,
        similarStudiesFiltersUINovel,
        similarStudiesFiltersNovel,
        similarStudiesFiltersUI,
        similarStudiesFilters,
        selectedStudiesSitesTab,
        selectedStudySimilarStudies,
        filteredSimilarStudies,
        candidateSiteFiltersUI,
        candidateSitesFilters,
        finalCandidateSites,
        openThirdTab,
        candidateSitesTableData,
        groupableCandidateSites,
        geoJsonSitesData
    }

})