import * as turf from '@turf/turf';
import { omit } from 'lodash';


export const geoPointStrToGeoJson = (geoPointWithProps: { [key: string]: any }, geoPointKey: string): turf.AllGeoJSON | null => {
    if (geoPointWithProps[geoPointKey] != null) {
        const coordinates = geoPointWithProps[geoPointKey]
            .replace('POINT(', '')
            .replace(')', '')
            .split(' ')
            .map(parseFloat);

        return turf.point(coordinates, { ...omit(geoPointWithProps, geoPointKey) });
    }

    return null;
}

export const getGeoJson = (geoPointsWithProps: { [key: string]: any }[], geoPointKey: string): turf.FeatureCollection => {
    return {
        type: 'FeatureCollection',
        // @ts-ignore
        features: geoPointsWithProps.map((value) => geoPointStrToGeoJson(value, geoPointKey)).filter((value) => value != null)
    }
}


export function extractLatLongFromGeoPoint(geoPointString: string | null): [number, number] | null {
    // Use a nullish coalescing operator to provide an empty array if no match is found
    if (!geoPointString) return null;
    const matches = geoPointString.match(/[-\d.]+/g) ?? [];

    // Ensure at least two values are present (longitude and latitude)
    if (matches.length >= 2) {
        const point = turf.point(matches.map(Number));

        // Extract latitude and longitude
        const coordinates = turf.getCoord(point);
        const [longitude, latitude] = coordinates;

        // Ensure the extracted values are valid numbers
        if (!isNaN(latitude) && !isNaN(longitude)) {
            return [latitude, longitude];
        }
    }

    // Return null if the values are not valid
    return null;
}

export function extractLatLongFromGeoPoints(geoPointStrings: string[]): [number, number][] {
    // @ts-ignore
    return geoPointStrings.map((value) => extractLatLongFromGeoPoint(value)).filter((value) => value != null);
}