import dataiku
import dash
from dash import dcc, html, dash_table
from dash.dependencies import Input, Output, State, ALL
import sklearn
import io
import pickle
import datetime
from flask import request
from main_functions import *
from layout_components_style import *
import dash_bootstrap_components as dbc

project = dataiku.api_client().get_default_project()
table_styles = get_table_styles()
button_style = get_button_style()
button_style_space = {**button_style, "margin-right": "7px"}
card_style = get_card_style()
selected_card_style = get_selected_card_style()
enter_box_style = get_enter_box_style()
tooltip_style = get_tooltip_style()
question_mark_style = get_question_mark_style()
header_h2_style = get_header_h2_style()
subheader_style = get_subheader_style()
description_style = get_description_style()
main_style = get_main_style()
left_panel_style = get_left_panel_style()
right_panel_style = get_right_panel_style()
dropdown_style = get_dropdown_style()
hidden_style = {'display': 'none'}
visible_style = {}
visible_tab_style = get_visible_tab_style()
card_margin = {'margin': '4px 0', 'padding': '1'}
tab_style = get_tab_style()
tab_selected_style = get_tab_selected_style()
div_message_style = get_div_message_style()
refresh_button_style = get_refresh_button_style()

  
# -----------------------
# Navigation (same pattern as previous webapp)
# -----------------------
TAB_ORDER = ['tab-1', 'tab-2']  # order of right/left tabs

def right_tab_nav(tab_value):
    """Prev/Next controls shown inside each right tab (Next hidden on last tab)."""
    idx = TAB_ORDER.index(tab_value)
    has_prev = idx > 0
    has_next = idx < len(TAB_ORDER) - 1  # last tab won't show Next

    return html.Div(
        [
            html.Button("◀ Previous", id=f"nav-prev-{tab_value}", n_clicks=0,
                        style={**button_style_space, 'visibility': 'visible' if has_prev else 'hidden'}),
            html.Button("Next ▶", id=f"nav-next-{tab_value}", n_clicks=0,
                        style={**button_style_space, 'visibility': 'visible' if has_next else 'hidden'}),
           # dbc.Tooltip("Go to the previous step.",
            #            target=f"nav-prev-{tab_value}", placement="left", style=tooltip_style),
            #dbc.Tooltip("Go to the next step.",
             #           target=f"nav-next-{tab_value}", placement="right", style=tooltip_style),
        ],
        style={'display': 'flex', 'justifyContent': 'flex-end', 'gap': '8px', 'margin': '10px 0'}
    )

# Fixed-position RESTART button in top-right corner
restart_button_div = html.Div([
    html.Button("REFRESH", id="clear-icon", style={**button_style, 'padding': '4px 10px'}),
    dbc.Tooltip(
        "Reset the interface to its default state. This won't delete any saved sessions.",
        target="clear-icon",
        placement="left",
        style=tooltip_style
    ),
    html.Div(id='refresh-trigger', style={'display': 'none'})
], style=refresh_button_style)

app.layout = html.Div([
    # Page Title
    restart_button_div,
    delete_confirmation_modal(),

    # Main container for left and right panels
    html.Div([
        # Left Panel: Tabs for the left panel
        html.Div([
            dcc.Tabs(id='left-panel-tabs', value='tab-1', children=[
                dcc.Tab(label='', value='tab-1', children=[
                    # Left Panel Content for 'Segmentation Session'
                    html.Div([
                        html.H2(['Select Segmentation',
                                 html.Span('?', id='tooltip-select-session', style=question_mark_style)
                                 ], style=header_h2_style),
                        dbc.Tooltip(
                            "Choose one of the existing segmentation to view the different session version and update segmentation results. Make sure that the updated dataset exists in the dataiku flow with the original dataset name as shown in the card.",
                            target="tooltip-select-session",
                            placement="right",
                            style=tooltip_style
                        ),
                        dcc.Dropdown(
                            id='session-name-dropdown',
                            placeholder="Search sessions",
                            searchable=True,
                            style=dropdown_style
                        ),
                        html.Div(
                            id='card-container',
                            className='card-container',
                            style={
                                'height': '900px',
                                'overflow-y': 'scroll',
                                'border': '1px solid #ddd',
                                'padding': '10px',
                                'background-color': 'white'
                            }
                        ),
                        dcc.Store(id='selected-card-index'),
                    ])
                ], style=hidden_style, selected_style=visible_tab_style),

                dcc.Tab(label='', value='tab-2', children=[
                    # Left Panel Content for 'Results'
                    html.Div([
                        html.H2(["Segmentation Processing",
                                 html.Span('?', id='tooltip-segments-process-update', style=question_mark_style)],
                                style=header_h2_style),
                        dbc.Tooltip(
                            "If you want to SAVE this updated segmentation session you must provide a name and segmentation description. These information are particularly useful metadata for future updates.",
                            target="tooltip-segments-process-update",
                            placement="right",
                            style=tooltip_style
                        ),
                        dcc.Textarea(
                            id='metadata-description',
                            placeholder='Enter a segmentation description..',
                            style=description_style
                        ),
                        html.Div([
                            html.Button("Save", id="save-icon", style=button_style_space),
                            dbc.Tooltip(
                                "SAVE button will create a csv file with the new Segmentation Results in the output_folder and a new record in the metadata_dataset with your updated session information.",
                                target="save-icon",
                                placement="bottom",
                                style=tooltip_style
                            ),
                            html.Button("Export", id="export-icon", style=button_style_space),
                            dbc.Tooltip(
                                "EXPORT button will download a csv file with the segmentation resuts locally.",
                                target="export-icon",
                                placement="bottom",
                                style=tooltip_style
                            ),
                            html.Span('?', id='tooltip-buttons-update', style=question_mark_style)
                        ]),
                        dbc.Tooltip(
                            "Hover over the buttons to see functionality details.",
                            target="tooltip-buttons-update",
                            placement="right",
                            style=tooltip_style
                        ),
                        dcc.Download(id="download-dataframe-csv"),
                        html.Div(id='save-output'),
                        dcc.Store(id='description-dictionary'),
                    ])
                ], style=hidden_style, selected_style=visible_tab_style)
            ])
        ], style=left_panel_style),

        # Right Panel: Tabs for the right panel
        html.Div([
            dcc.Tabs(
                id='right-panel-tabs',
                value='tab-1',
                children=[
                    dcc.Tab(
                        label='1. Segmentation Session',
                        value='tab-1',
                        style=tab_style,
                        selected_style=tab_selected_style,
                        children=[
                            html.Div([
                                right_tab_nav('tab-1'),  # ← navigation row
                                html.Div(id='selected-session-name',
                                         style={'font-weight': 'bold', 'margin-bottom': '10px'}),
                                html.Div(id='file-content-container',
                                         children=[
                                             dash_table.DataTable(
                                                 id='file-content-table',
                                                 columns=[],
                                                 data=[],
                                                 filter_action='native',
                                                 sort_action='native',
                                                 page_size=5,
                                                 **table_styles
                                             ),
                                             html.Div(id='file-content-error',
                                                      style={'color': 'black', 'margin-top': '10px'})
                                         ]),
                                html.Div([
                                    html.Div([
                                        html.Button('Update', id='update-button',
                                                    style={**button_style, 'display': 'none', 'margin-right': '10px'}),
                                        dbc.Tooltip(
                                            "Once you select the UPDATE button, the same filtering, segmentation method, and feature selection displayed on the selected card will be applied to the updated original dataset. The updated version won't be saved unless you select SAVE in the Insights tab.",
                                            target="update-button",
                                            placement="bottom",
                                            style=tooltip_style
                                        ),
                                        html.Button('Delete', id='delete-button',
                                                    style={**button_style, 'display': 'none', 'margin-right': '10px'}),
                                        dbc.Tooltip(
                                            "DELETE button will permanently delete ALL the sessions of the Selected Segmentation and restart the webapp. Select responsibly!",
                                            target="delete-button",
                                            placement="bottom",
                                            style=tooltip_style
                                        ),
                                        html.Button('Make Active', id='activate-button',
                                                    style={**button_style, 'display': 'none', 'margin-right': '10px'}),
                                        dbc.Tooltip(
                                            "MAKE ACTIVE button will set this version as the active one, making it the default version used in downstream processes.",
                                            target="activate-button",
                                            placement="bottom",
                                            style=tooltip_style
                                        ),
                                        html.Span('?', id='tooltip-update-button',
                                                  style={**question_mark_style, 'display': 'none'}),
                                        dbc.Tooltip(
                                            "The table above shows the results of the last segmentation version and dataset. Once 'UPDATE' is selected, a new table and a Sankey graph below will display the updated segmentation results, allowing you to compare the changes in account segmentations. Note that table includes only the records that there was a cluster change. Sankey graph and Results in the next tab process the entire updated dataset.",
                                            target="tooltip-update-button",
                                            placement="right",
                                            style=tooltip_style
                                        ),
                                    ], style={'display': 'flex', 'align-items': 'center', 'margin-bottom': '10px'}),
                                    # Stores
                                    dcc.Store(id='versioning-store'),
                                    dcc.Store(id='max-versioning-store'),
                                    dcc.Store(id='status-store'),
                                    dcc.Store(id='delete-timestamp-store'),
                                    dcc.Store(id='activate-timestamp-store'),
                                    dcc.Store(id='update-timestamp-store'),
                                    dcc.Store(id='message-reset-signal', data=0),
                                    # Messages
                                    html.Div([
                                        html.Div(id='activate-message',
                                                 style={'color': 'black', 'margin-top': '10px'}),
                                        html.Div(id='delete-warning',
                                                 style={'color': 'black', 'margin-top': '10px'}),
                                        dcc.Loading(
                                            id="loading-delete-message",
                                            type="circle",
                                            children=html.Div(id='delete-message'),
                                            fullscreen=False
                                        )
                                    ]),
                                    html.Div(id='update-message',
                                             style={'color': 'black', 'margin-top': '10px'}),
                                ]),
                                html.Div([
                                    dcc.Loading(
                                        id="loading-data-upload",
                                        type="circle",
                                        children=[
                                            html.Div([
                                                html.Div(
                                                    dash_table.DataTable(
                                                        id='output-table',
                                                        columns=[],
                                                        data=[],
                                                        filter_action='native',
                                                        sort_action='native',
                                                        page_size=10,
                                                        **table_styles
                                                    ),
                                                    id='table-container',
                                                    style={'display': 'none'}
                                                ),
                                                html.Div(
                                                    id="no-data-message",
                                                    style={'textAlign': 'center', 'fontSize': 12, 'color': 'gray'}
                                                ),
                                                dcc.Input(
                                                    id='file-name-input',
                                                    type='text',
                                                    placeholder='Enter a file name',
                                                    style={**enter_box_style, 'margin-top': '20px',
                                                           'width': '100%', 'display': 'none'}
                                                ),
                                                html.Button('Export Comparison', id='export-comp-button',
                                                            style={**button_style, 'display': 'none',
                                                                   'margin-right': '10px', 'margin-top': '10px'}),
                                                html.Div(id='export-message',
                                                         style={'margin-top': '20px', 'display': 'none'}),
                                            ]),
                                        ],
                                    ),
                                    html.Div(id='sankey-container', children=[
                                        dcc.Graph(id='sankey-diagram')
                                    ], style={'width': '69%', 'display': 'none', 'padding-left': '10px'}),
                                ], style={'display': 'flex', 'justify-content': 'space-between'}),
                                # Storage elements for state management
                                dcc.Store(id='selected-features-store'),
                                dcc.Store(id='session-name-store'),
                                dcc.Store(id='original-dataset-store'),
                                dcc.Store(id='method-store'),
                                dcc.Store(id='specifications-store'),
                                dcc.Store(id='remap-dict-store'),
                                dcc.Store(id='full-data-store'),
                                dcc.Store(id='importance-data-store')
                            ])
                        ]
                    ),
                    dcc.Tab(
                        label='2. Insights',
                        value='tab-2',
                        style=tab_style,
                        selected_style=tab_selected_style,
                        children=[
                            html.Div([
                                right_tab_nav('tab-2'),  # ← navigation row (Next hidden)
                                html.Div([
                                    html.H2(['Segmentation Insights',
                                             html.Span('?', id='tooltip-segmentation-insights-update',
                                                       style=question_mark_style)
                                             ], style=header_h2_style),
                                    dbc.Tooltip(
                                        "Once you select and UPDATE a segmentation session, the graphs below will support the explainability of your updated results.",
                                        target="tooltip-segmentation-insights-update",
                                        placement="right",
                                        style=tooltip_style
                                    ),
                                    html.Div([
                                        html.Div([
                                            html.H3("Number of Records in Each Segment", style=subheader_style),
                                            dcc.Graph(id='cluster-histogram'),
                                        ], style={'width': '49%', 'display': 'inline-block'}),
                                        html.Div([
                                            html.H3(['Variable Importance',
                                                     html.Span('?', id='tooltip-variable-importance',
                                                               style=question_mark_style)
                                                     ], style=subheader_style),
                                            dbc.Tooltip(
                                                "For machine learning, variable importance identifies which features most influence the formation of segments. Higher values (closer to 1) mean the feature contributed more to distinguishing between clusters and lower values (near 0) mean the feature had little impact in splitting the data for classification. In rule-based segmentation, feature weights directly control how much each feature contributes to the segmentation outcome.",
                                                target="tooltip-variable-importance",
                                                placement="right",
                                                style=tooltip_style
                                            ),
                                            dcc.Graph(id='importance-histogram'),
                                        ], style={'width': '49%', 'display': 'inline-block', 'float': 'right'}),
                                    ]),
                                    html.H2("Feature Distribution Analysis", style=header_h2_style),
                                    html.Div([
                                        html.Div([
                                            html.H3("Variable Distribution by Segment", style=subheader_style),
                                            dcc.Dropdown(id='feature-dropdown',
                                                         placeholder="Select a feature to display",
                                                         style=dropdown_style),
                                            html.Div(id="feature-histograms"),
                                        ], style={'width': '49%', 'display': 'inline-block'}),
                                        html.Div([
                                            html.H3(["Average Feature Values by Segment",
                                                     html.Span('?', id='tooltip-heatmap', style=question_mark_style)],
                                                    style=subheader_style),
                                            dbc.Tooltip(
                                                "Compare how each selected numerical feature differs across segments. Darker colors indicate relatively higher averages in that segment. Hover over a cell to view the actual average value.",
                                                target="tooltip-heatmap",
                                                placement="right",
                                                style=tooltip_style
                                            ),
                                            dcc.Dropdown(id='numerical-feature-multiselect', multi=True,
                                                         placeholder="Select numerical features",
                                                         style=dropdown_style),
                                            html.Div(id='mean-heatmap'),
                                        ], style={'width': '49%', 'display': 'inline-block', 'float': 'right'}),
                                    ]),
                                    html.Div(id="message-div", style={"color": "black", "font-weight": "bold"})
                                ], style={'width': '100%', 'padding': '20px'})
                            ])
                        ]
                    )
                ],
                style={
                    'display': 'flex',
                    'justify-content': 'flex-start',
                    'margin': '0px 5px',
                    'border': 'none'
                }
            ),
        ], style=right_panel_style)
    ], style=main_style)  # Flexbox for main container
])

# ------------------------------------
# Sync callback (unchanged)
# ------------------------------------
@app.callback(
    [Output('left-panel-tabs', 'value'), Output('right-panel-tabs', 'value')],
    [Input('left-panel-tabs', 'value'), Input('right-panel-tabs', 'value')]
)
def sync_tabs(left_tab, right_tab):
    ctx = dash.callback_context
    if not ctx.triggered:
        return dash.no_update, dash.no_update
    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]
    if triggered_id == 'left-panel-tabs':
        return left_tab, left_tab
    else:
        return right_tab, right_tab

# ------------------------------------
# Navigation callback (Prev/Next inside right tabs)
# Note: Only updates LEFT tab value; sync mirrors it right.
# allow_duplicate=True avoids duplicate-output error.
# ------------------------------------
@app.callback(
    Output('left-panel-tabs', 'value', allow_duplicate=True),
    [
        # Prev buttons
        Input('nav-prev-tab-1', 'n_clicks'),
        Input('nav-prev-tab-2', 'n_clicks'),
        # Next buttons
        Input('nav-next-tab-1', 'n_clicks'),
        Input('nav-next-tab-2', 'n_clicks'),
    ],
    State('right-panel-tabs', 'value'),
    prevent_initial_call=True
)
def navigate_tabs(prev_t1, prev_t2, next_t1, next_t2, current_right):
    ctx = dash.callback_context
    if not ctx.triggered:
        raise dash.exceptions.PreventUpdate

    button_id = ctx.triggered[0]['prop_id'].split('.')[0]
    order = TAB_ORDER
    idx = order.index(current_right)

    # Previous
    if button_id == 'nav-prev-tab-2':
        return order[idx - 1] if idx > 0 else order[0]
    if button_id == 'nav-prev-tab-1':
        return order[0]  # first tab prev (hidden), safe-guard

    # Next
    if button_id == 'nav-next-tab-1':
        return order[idx + 1] if idx < len(order) - 1 else order[-1]
    if button_id == 'nav-next-tab-2':
        return order[-1]  # last tab next is hidden; safe-guard

    raise dash.exceptions.PreventUpdate

# -------------------------
# Rest of your original callbacks (unchanged)
# -------------------------

@app.callback(
    Output('session-name-dropdown', 'options'),
    [Input('delete-timestamp-store', 'data'),
     Input('activate-timestamp-store', 'data'),
     Input('update-timestamp-store', 'data')]
)
def update_dropdown_options(delete_ts, activate_ts, update_ts):
    metadata = dataiku.Dataset("metadata_dataset")
    metadata_df = metadata.get_dataframe().reset_index(drop=True)
    session_names = sorted(metadata_df['session_name'].dropna().unique())
    return [{'label': name, 'value': name} for name in session_names]


@app.callback(
    Output('card-container', 'children'),
    [Input('session-name-dropdown', 'value'),
     Input('delete-timestamp-store', 'data'),
     Input('activate-timestamp-store', 'data'),
     Input('update-timestamp-store', 'data')],
)
def update_cards(selected_session_name, delete_ts, activate_ts, update_ts):
    metadata = dataiku.Dataset("metadata_dataset")
    metadata_df = metadata.get_dataframe().reset_index(drop=True)
    print('cards')
    print(metadata_df)
    if not selected_session_name:
        return []

    filtered_df = metadata_df[metadata_df['session_name'] == selected_session_name]
    filtered_df['version_name'] = filtered_df['session_name'] + "_" + filtered_df['segmentation_version'].astype(str)
    print(filtered_df)
    cards = []
    for i, row in filtered_df.iterrows():
        version_name = row['version_name']
        card = html.Div([
            html.Button(
                id={'type': 'card-button', 'index': i},
                children=html.Div([
                    html.Label(f"{version_name} ({row['status']})", style={'font-weight': 'bold'}),
                    html.P(f"Dataset: {row['original_dataset']}", style=card_margin),
                    html.P(f"User: {row['username']}", style=card_margin),
                    html.P(f"Date: {row['datetime']}", style=card_margin),
                    html.P(f"Method: {row['method']}", style=card_margin),
                ]),
                style=card_style,
                className='hover-effect'
            ),
            dbc.Tooltip(
                html.Div([
                    f"Description: {row['description']}",
                    html.Br(),
                    f"Features: {row['selected_features']}"
                ]),
                target={'type': 'card-button', 'index': i},
                placement="right",
                style=tooltip_style
            )
        ], className='mini-card')
        cards.append(card)
    return cards


@app.callback(
    Output({'type': 'card-button', 'index': ALL}, 'style'),
    Input('selected-card-index', 'data'),
    State({'type': 'card-button', 'index': ALL}, 'id')
)
def update_card_selection(selected_index, all_ids):
    if selected_index is None:
        return [card_style] * len(all_ids)

    return [
        selected_card_style if btn['index'] == selected_index else card_style
        for btn in all_ids
    ]


@app.callback(
    [Output('selected-session-name', 'children'),
     Output('file-content-container', 'children'),
     Output('update-button', 'style'),
     Output('tooltip-update-button', 'style'),
     Output('delete-button', 'style'),
     Output('activate-button', 'style'),
     Output('versioning-store', 'data'),
     Output('status-store', 'data'),
     Output('message-reset-signal', 'data'),
     Output('selected-card-index', 'data')],
    [Input({'type': 'card-button', 'index': dash.dependencies.ALL}, 'n_clicks'),
     Input('delete-timestamp-store', 'data'),
     Input('activate-timestamp-store', 'data'),
     Input('update-timestamp-store', 'data'),
     ]
)
def select_session(n_clicks, delete_timestamp, activate_timestamp, update_timestamp):
    ctx = dash.callback_context
    triggered_id = ctx.triggered_id

    if triggered_id in ['delete-timestamp-store', 'activate-timestamp-store', 'update-timestamp-store']:
        action = triggered_id.split('-')[0].capitalize()
        return ('', html.Div(html.P(f"Please select a session.", style=div_message_style)),
                hidden_style, hidden_style, hidden_style, hidden_style, None, None,
                datetime.now().timestamp(), dash.no_update)

    if not n_clicks or all(click is None or click == 0 for click in n_clicks):
        return ('', html.Div(html.P("No data selected yet.", style=div_message_style)),
                hidden_style, hidden_style, hidden_style, hidden_style, None, None, dash.no_update, dash.no_update)

    if not triggered_id or 'index' not in triggered_id:
        return ('', '', hidden_style, hidden_style, hidden_style, hidden_style, None, None, dash.no_update, dash.no_update)

    clicked_index = triggered_id['index']

    metadata = dataiku.Dataset("metadata_dataset")
    metadata_df = metadata.get_dataframe().reset_index(drop=True)
    print("select")
    print(metadata_df)

    if clicked_index >= len(metadata_df):
        return ('', html.Div(html.P("Error: Selected session is no longer available. Please select another session.",
                                    style=div_message_style)),
                hidden_style, hidden_style, hidden_style, hidden_style, None, None,
                datetime.now().timestamp(), dash.no_update)

    session_name = metadata_df['session_name'].iloc[clicked_index]
    version = metadata_df['segmentation_version'].iloc[clicked_index]
    status = metadata_df['status'].iloc[clicked_index]
    selected_features, segmentation_method, specifications, cluster_remap_dict, original_dataset_name = \
        load_segmentation_parameters(metadata_df, clicked_index)
    print(session_name, version, status)
    reset_signal = datetime.now().timestamp()

    if original_dataset_name not in [ds.name for ds in project.list_datasets()]:
        return (f'Selected Session: {session_name}',
                html.Div([
                    html.P(f"Error: Dataset '{original_dataset_name}' not found in Dataiku Flow. Please make sure the original dataset exists."),
                ], style=div_message_style), hidden_style, hidden_style, hidden_style, hidden_style, None, None,
                datetime.now().timestamp()), dash.no_update

    columns, data = display_file_content(session_name)

    if columns is None:
        return (f'Selected Session: {session_name}',
                html.Div(data, style={'color': 'black', 'margin-top': '10px'}),
                hidden_style, hidden_style, hidden_style, hidden_style,
                version, status, reset_signal, dash.no_update)

    table_component = dash_table.DataTable(
        id='file-content-table',
        columns=columns,
        data=data,
        filter_action='native',
        sort_action='native',
        page_size=5,
        **table_styles
    )

    return (f'Selected Session: {session_name}',
            table_component,
            {**button_style_space, 'display': 'block'},
            {**question_mark_style, 'display': 'inline-block'},
            {**button_style_space, 'display': 'block'},
            {**button_style, 'display': 'block'},
            version, status, reset_signal, clicked_index)


@app.callback(
    [Output('table-container', 'style'),
     Output('no-data-message', 'children'),
     Output('no-data-message', 'style'),
     Output('output-table', 'columns'),
     Output('output-table', 'data'),
     Output('update-message', 'children'),
     Output('session-name-store', 'data'),
     Output('selected-features-store', 'data'),
     Output('feature-dropdown', 'options'),
     Output('original-dataset-store', 'data'),
     Output('method-store', 'data'),
     Output('specifications-store', 'data'),
     Output('remap-dict-store', 'data'),
     Output('max-versioning-store', 'data'),
     Output('sankey-container', 'style'),
     Output('sankey-diagram', 'figure'),
     Output('full-data-store', 'data'),
     Output('importance-data-store', 'data'),
     Output('file-name-input', 'style'),
     Output('export-comp-button', 'style')],
    [Input('update-button', 'n_clicks'),
     Input('message-reset-signal', 'data')],
    [State('selected-session-name', 'children'),
     State('file-content-table', 'data'),
     State('versioning-store', 'data'),
     State('status-store', 'data')]
)
def update_segmentation(n_clicks, reset_signal, selected_session_text, file_content_data, version, status):
    ctx = dash.callback_context
    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]
    sankey_container_style = {'display': 'none'}
    default_clear_return = hidden_style, '', visible_style, [], [], '', dash.no_update, None, None, None, None, None, None, None, sankey_container_style, create_empty_figure(), None, None, hidden_style, hidden_style

    if triggered_id == 'message-reset-signal' and reset_signal:
        return default_clear_return

    if n_clicks is None or selected_session_text == '':
        return default_clear_return

    metadata = dataiku.Dataset("metadata_dataset")
    metadata_df = metadata.get_dataframe().reset_index(drop=True)
    session_name = selected_session_text.replace('Selected Session: ', '')
    metadata_vers = metadata_df[(metadata_df['session_name'] == session_name)]
    max_seg_version = metadata_vers['segmentation_version'].max()
    metadata_session = metadata_df[(metadata_df['session_name'] == session_name) &
                                   (metadata_df['segmentation_version'] == version)]

    if status == "inactive":
        return hidden_style, html.P("Warning: You cannot update an inactive session. Please make the session active first.",
                                    style=div_message_style), visible_style, [], [], '', dash.no_update, None, None, None, None, None, None, None, sankey_container_style, create_empty_figure(), None, None, hidden_style, hidden_style

    selected_features, segmentation_method, specifications, cluster_remap_dict, original_dataset_name = \
        load_segmentation_parameters(metadata_session, 0)
    segmentation_version = version

    if original_dataset_name not in [ds.name for ds in project.list_datasets()]:
        error_message_data = f"Error: dataset '{original_dataset_name}' does not exist in the Dataiku flow. Please check the name of existing datasets to align with the selected sessions."
        return hidden_style, html.P(error_message_data, style=div_message_style), visible_style, [], [], '', dash.no_update, None, None, None, None, None, None, None, sankey_container_style, create_empty_figure(), None, None, hidden_style, hidden_style

    original_dataset = dataiku.Dataset(original_dataset_name)
    original_dataset_df = original_dataset.get_dataframe()

    if all(col in original_dataset_df.columns for col in selected_features):
        if not bool(specifications):
            filtered_df = original_dataset_df.copy()
        else:
            filtered_df = apply_filters(original_dataset_df, specifications)

        if filtered_df.empty:
            message_filt = "Warning: Filtered data is empty. Please check the original input dataset and try again."
            return hidden_style, html.P(message_filt, style=div_message_style), visible_style, [], [], "", dash.no_update, None, None, None, None, None, None, None, sankey_container_style, create_empty_figure(), None, None, hidden_style, hidden_style

        if file_content_data:
            df_file_content = pd.DataFrame(file_content_data)

        cluster_column_name = 'cluster'
        if segmentation_method == "kmeans":
            model_folder_id = output_model_folder_var
            df_results, preprocessor = apply_existing_model(cluster_column_name, filtered_df, model_folder_id,
                                                            session_name, cluster_remap_dict)
            print("HEREEEE")
            print(df_results)
            feature_importance_df = feature_importance_rfclassifier(df_results, selected_features, preprocessor,
                                                                    cluster_column_name)
        else:
            df_results, weights, warning_message = apply_existing_bounds(session_name, selected_features,
                                                                         df_file_content, filtered_df,
                                                                         cluster_remap_dict)
            if warning_message:
                return hidden_style, html.P(warning_message, style=div_message_style), visible_style, [], [], '', dash.no_update, None, None, None, None, None, None, None, sankey_container_style, create_empty_figure(), None, None, hidden_style, hidden_style
            importance_dict = weights
            feature_importance_df = pd.DataFrame(list(importance_dict.items()),
                                                 columns=['Feature', 'Importance']).sort_values(by='Importance',
                                                                                                ascending=False)

        if file_content_data:
            df_file_content = df_file_content[['account_id', 'cluster']].rename(columns={'cluster': 'old_cluster'})
            df_results = pd.merge(df_results, df_file_content, on='account_id', how='outer')

        cluster_id_columns = ['account_id', 'old_cluster', 'cluster']
        feature_options = generate_feature_options(df_results, cluster_id_columns)

        df_results_filtered = df_results[cluster_id_columns]
        df_results_filtered = df_results_filtered.rename(columns={'old_cluster': 'old_version',
                                                                  'cluster': 'new_segmentation'})

        df_results_changed = df_results_filtered[df_results_filtered['old_version'] != df_results_filtered['new_segmentation']]

        columns = [{"name": i, "id": i} for i in df_results_changed.columns]
        data = df_results_changed.to_dict('records')

        if df_results_changed.empty:
            sankey_fig = create_empty_figure()
            update_message = f"Segmentation results for {session_name} are the same."
            table_style = hidden_style
            export_comp_button = hidden_style
            enter_text = hidden_style
        else:
            sankey_fig = create_sankey(df_results_filtered, 'old_version', 'new_segmentation')
            update_message = " "
            sankey_container_style = {'display': 'block'}
            table_style = visible_style
            export_comp_button = {**button_style, **visible_style, 'margin-top': '10px'}
            enter_text = {**enter_box_style, **visible_style}

        df_results_new = df_results.drop(columns=['old_cluster'])
        result_cols = df_results_new.columns.tolist()
        result_cols.insert(result_cols.index('account_id') + 1, result_cols.pop(result_cols.index('cluster')))
        df_results_new = df_results_new[result_cols]
        full_data_store = df_results_new.to_dict('records')
        
        restart_app_silently(multi_seg_explorer_webapp_id)

        return (table_style, "", hidden_style, columns, data,
                html.P(update_message, style=div_message_style),
                session_name, selected_features, feature_options, original_dataset_name,
                segmentation_method, specifications, cluster_remap_dict, max_seg_version,
                sankey_container_style, sankey_fig, full_data_store, feature_importance_df.to_dict('records'),
                enter_text, export_comp_button)

    else:
        message_warn = f"Warning: Not all columns used for the original segmentation exist in the updated dataset. Check '{selected_features}'"
        return hidden_style, html.P(message_warn, style=div_message_style), visible_style, [], [], '', dash.no_update, None, None, None, None, None, None, None, sankey_container_style, create_empty_figure(), None, None, hidden_style, hidden_style


@app.callback(
    [Output('export-message', 'children'),
     Output('export-message', 'style')],
    [Input('export-comp-button', 'n_clicks'),
     Input('message-reset-signal', 'data')],
    [State('file-name-input', 'value'),
     State('output-table', 'data')]
)
def export_data(n_clicks, reset_signal, file_name, data):
    ctx = dash.callback_context
    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]

    if triggered_id == 'message-reset-signal':
        return '', hidden_style

    if not n_clicks:
        return '', hidden_style

    if not file_name or file_name.strip() == '':
        return html.P('Please enter a valid file name before exporting.'), {**div_message_style, 'display': 'block'}

    if not data:
        return html.P('No data available to export.'), {**div_message_style, 'display': 'block'}

    df = pd.DataFrame(data)
    compare_filename = f"{file_name.strip()}.csv"
    compare_seg_folder.upload_data(compare_filename, df.to_csv(index=False).encode("utf-8"))

    return html.P('Data have been successfully saved in the compare sessions folder!'), {**div_message_style, 'display': 'block'}


@app.callback(
    [Output("delete-confirmation-modal", "is_open"),
     Output("delete-warning", "children")],
    [Input("delete-button", "n_clicks"),
     Input("cancel-delete", "n_clicks"),
     Input("confirm-delete", "n_clicks")],
    [State("status-store", "data"),
     State("selected-session-name", "children"),
     State("delete-confirmation-modal", "is_open")]
)
def toggle_delete_modal(delete_click, cancel_click, confirm_click, status, session_name_raw, is_open):
    ctx = dash.callback_context
    if not ctx.triggered:
        return is_open, ""

    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]

    if triggered_id == "delete-button":
        if status == "inactive":
            return True, ""  # Show modal

        session_name = session_name_raw.replace("Selected Session: ", "")
        metadata = dataiku.Dataset("metadata_dataset")
        metadata_df = metadata.get_dataframe()
        session_versions = metadata_df[metadata_df['session_name'] == session_name]

        if len(session_versions) == 1 and session_versions.iloc[0]['status'] == "active":
            return True, html.P(
                "Note: This is the only version of this session and is currently active. "
                "Deleting will remove all associated data.",
                style=div_message_style
            )

        return False, html.P(
            "Warning: Cannot delete an active session unless it is the only version available.",
            style=div_message_style
        )
    elif triggered_id in ["cancel-delete", "confirm-delete"]:
        return False, ""  # Hide modal and clear message

    return is_open, ""


@app.callback(
    [Output('delete-message', 'children'),
     Output('delete-timestamp-store', 'data')],
    [Input('confirm-delete', 'n_clicks'),
     Input('message-reset-signal', 'data')],
    [State('selected-session-name', 'children'),
     State('versioning-store', 'data'),
     State('status-store', 'data')]
)
def delete_segmentation(n_clicks, reset_signal, selected_session_name, version, status):
    ctx = dash.callback_context
    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]

    if triggered_id == 'message-reset-signal' and reset_signal:
        return ["", dash.no_update]

    if not n_clicks:
        return ["", None]

    try:
        session_name = selected_session_name.replace('Selected Session: ', '')
        print(f"Attempting to delete: {session_name} (Version: {version}, Status: {status})")

        metadata = dataiku.Dataset("metadata_dataset")
        metadata_df = metadata.get_dataframe().reset_index(drop=True)

        version_exists = ((metadata_df['session_name'] == session_name) &
                          (metadata_df['segmentation_version'] == version)).any()
        if not version_exists:
            return [html.P(f"Error: Version {version} of {session_name} no longer exists.", style=div_message_style), dash.no_update]

        session_versions = metadata_df[metadata_df['session_name'] == session_name]
        method = session_versions[session_versions['segmentation_version'] == version]['method'].values[0]

        if status == "active" and len(session_versions) > 1:
            return [html.P("Cannot delete an active session while other versions exist.", style=div_message_style), dash.no_update]

        metadata_del = metadata_df[
            ~((metadata_df['session_name'] == session_name) & (metadata_df['segmentation_version'] == version))
        ].reset_index(drop=True)

        version_filename = f"/{session_name}_{version}.csv"
        version_files = versions_folder.list_paths_in_partition()
        version_deleted = False
        if version_filename in version_files:
            versions_folder.delete_path(version_filename)
            version_deleted = True
            print(f"Deleted from versions folder: {version_filename}")

        output_deleted = False
        if len(session_versions) == 1:
            output_filename = f"/{session_name}.csv"
            output_files = output_folder.list_paths_in_partition()
            if output_filename in output_files:
                output_folder.delete_path(output_filename)
                output_deleted = True
                print(f"Deleted from output folder: {output_filename}")

        if method == "rule_based":
            try:
                specs_dataset = dataiku.Dataset("rule_based_specs_dataset")
                specs_df = specs_dataset.get_dataframe()
                updated_specs = specs_df[specs_df["session_name"] != session_name]
                specs_dataset.write_with_schema(updated_specs)
                print(f"Removed rule-based specs for: {session_name}")
            except Exception as e:
                print(f"Error updating rule_based_specs_dataset: {str(e)}")
        elif method == "kmeans":
            try:
                pkl_filename = f"/{session_name}.pkl"
                model_files = output_model_folder.list_paths_in_partition()
                if pkl_filename in model_files:
                    output_model_folder.delete_path(pkl_filename)
                    print(f"Deleted model file: {pkl_filename}")
            except Exception as e:
                print(f"Error deleting model file for kmeans: {str(e)}")

        metadata.write_with_schema(metadata_del)

        success_message = f"Version {version} of {session_name} has been successfully deleted!"
        restart_app_silently(multi_seg_explorer_webapp_id)
        
        if not version_deleted:
            success_message += " (Warning: Version file not found in versions folder.)"
        if len(session_versions) == 1 and not output_deleted:
            success_message += " (Warning: Output file not found.)"

        timestamp = datetime.now().timestamp()
        return [html.P(success_message, style=div_message_style), timestamp]

    except Exception as e:
        print(f"Error during deletion: {str(e)}")
        return [html.P(f"An error occurred while deleting: {str(e)}", style=div_message_style), dash.no_update]


@app.callback(
    [Output('activate-message', 'children'),
     Output('activate-timestamp-store', 'data')],
    [Input('activate-button', 'n_clicks'),
     Input('message-reset-signal', 'data')],
    [State('selected-session-name', 'children'),
     State('versioning-store', 'data'),
     State('status-store', 'data')]
)
def activate_segmentation(n_clicks, reset_signal, selected_session_name, version, status):
    ctx = dash.callback_context
    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]

    if triggered_id == 'message-reset-signal' and reset_signal:
        return ["", dash.no_update]

    if not n_clicks:
        return ["", None]

    try:
        session_name = selected_session_name.replace('Selected Session: ', '')
        metadata = dataiku.Dataset("metadata_dataset")
        metadata_df = metadata.get_dataframe()

        if status == "active":
            return [html.P(f"Warning: Version {session_name}_{version} is already active.", style=div_message_style), dash.no_update]

        metadata_df.loc[(metadata_df['session_name'] == session_name) & (metadata_df['status'] == "active"),
                        'status'] = "inactive"

        metadata_df.loc[(metadata_df['session_name'] == session_name) &
                        (metadata_df['segmentation_version'] == version), 'status'] = "active"
        metadata.write_with_schema(metadata_df)

        version_file = f"/{session_name}_{version}.csv"
        output_file = f"/{session_name}.csv"
        version_files = versions_folder.list_paths_in_partition()

        if version_file in version_files:
            with versions_folder.get_download_stream(version_file) as stream:
                content = stream.read().decode('utf-8')

            with output_folder.get_writer(output_file) as writer:
                writer.write(content.encode('utf-8'))
            restart_app_silently(multi_seg_explorer_webapp_id)
            timestamp = datetime.now().timestamp()
            return [html.P(f"Version {version} of {session_name} is now active!", style=div_message_style), timestamp]
            restart_app_silently(multi_seg_explorer_webapp_id)
        else:
            return [html.P(f"Error: {version_file} not found.", style=div_message_style), dash.no_update]

    except Exception as e:
        print(f"Error during activation: {str(e)}")
        return [html.P(f"An error occurred while activating: {str(e)}", style=div_message_style), dash.no_update]


@app.callback(
    Output("feature-histograms", "children"),
    [Input("feature-dropdown", "value")],
    [State("full-data-store", "data")]
)
def update_histograms_output(selected_feature, table_data):
    if not selected_feature or not table_data:
        return html.Div("", style=div_message_style)
    return update_mixed_graph(selected_feature, table_data)


@app.callback(
    Output('importance-histogram', 'figure'),
    [Input('importance-data-store', 'data')]
)
def update_importance_histogram(importance_data):
    try:
        if not importance_data:
            return create_empty_figure()

        importance_df = pd.DataFrame(importance_data)
        importance_df = importance_df.sort_values('Importance', ascending=False).head(10)
        return histogram_figure(importance_df, "Importance", "Feature",
                                {'Importance': 'Importance Score', 'Feature': 'Features'})
    except Exception as e:
        print(f"Error in importance histogram: {e}")
        return create_empty_figure()


@app.callback(
    Output('cluster-histogram', 'figure'),
    [Input('full-data-store', 'data')]
)
def update_cluster_pie_chart_output(table_data):
    if not table_data:
        return create_empty_figure()
    return update_cluster_pie_chart(table_data)


@app.callback(
    Output('numerical-feature-multiselect', 'options'),
    [Input('full-data-store', 'data')]
)
def update_numerical_feature_multiselect_output(table_data):
    return update_numerical_feature_multiselect(table_data)


@app.callback(
    Output('numerical-feature-multiselect', 'value'),
    Input('numerical-feature-multiselect', 'value'),
    State('numerical-feature-multiselect', 'options')
)
def handle_select_all_numerical_feature_multiselect(selected_values, available_options):
    if selected_values is None or available_options is None:
        return []
    if "select_all" in selected_values:
        return [option['value'] for option in available_options if option['value'] != 'select_all']
    return selected_values


@app.callback(
    Output('mean-heatmap', 'children'),
    [Input('numerical-feature-multiselect', 'value')],
    [State('full-data-store', 'data')]
)
def update_heatmap(selected_numerical_features, table_data):
    if table_data and selected_numerical_features:
        df = pd.DataFrame(table_data)
        return create_average_heatmap(df, selected_numerical_features, cluster_column_name='cluster')
    return None


@app.callback(
    Output("download-dataframe-csv", "data"),
    Input("export-icon", "n_clicks"),
    [State('session-name-store', 'data'),
     State("full-data-store", "data")],
    prevent_initial_call=True,
)
def export_data_as_csv(n_clicks, session_name, output_data):
    if n_clicks > 0 and output_data is not None:
        df = pd.DataFrame(output_data)
        return dcc.send_data_frame(df.to_csv, session_name + "_results.csv")
    return None


@app.callback(
    Output("description-dictionary", "data"),
    Input("metadata-description", "value")
)
def store_description(description):
    return {} if not description else {"description": description}


@app.callback(
    Output('save-output', 'children'),
    [Input('save-icon', 'n_clicks')],
    [State("full-data-store", "data"),
     State('remap-dict-store', 'data'),
     State('session-name-store', 'data'),
     State('max-versioning-store', 'data'),
     State('original-dataset-store', 'data'),
     State('method-store', 'data'),
     State('description-dictionary', 'data'),
     State('specifications-store', 'data'),
     State('selected-features-store', 'data')],
    prevent_initial_call=True
)
def save_data(save_clicks, output_data, cluster_remap_dict, session_name, max_segmentation_version,
              original_dataset_name, segmentation_method, description_dict, specifications, selected_features):
    ctx = dash.callback_context
    description = (description_dict or {}).get("description", "")
    print(cluster_remap_dict, session_name, max_segmentation_version, original_dataset_name,
          segmentation_method, description, specifications, selected_features)

    if not description:
        return html.P("Error: You must provide a description before saving the data.", style=div_message_style)

    if output_data and session_name and max_segmentation_version is not None:
        print("Proceeding with save")
        df = pd.DataFrame(output_data)
        updated_version = max_segmentation_version + 1
        status = 'active'

        metadata_record = create_metadata_record(
            session_name, updated_version, status,
            original_dataset_name, segmentation_method,
            description, specifications, selected_features,
            cluster_remap_dict
        )
        save_result_component = save_updated_data_to_dataiku(session_name, updated_version, df, metadata_record)

        restart_app_silently(multi_seg_explorer_webapp_id)
        
        return save_result_component

    return html.P("Failed to save data. Please check if all necessary data is available.", style=div_message_style)


@app.callback(
    Output('refresh-trigger', 'children'),
    Input('clear-icon', 'n_clicks'),
    prevent_initial_call=True
)
def trigger_clear(n_clicks):
    if n_clicks:
        return "REFRESH"
    return dash.no_update


app.clientside_callback(
    """
    function(trigger) {
        if (trigger === "REFRESH") {
            setTimeout(function() {
                window.location.reload();
            }, 500);
        }
        return '';
    }
    """,
    Output('refresh-trigger', 'style'),
    Input('refresh-trigger', 'children'),
    prevent_initial_call=True
)