import re
import dash
from dash import dcc, html, dash_table, Input, Output, State
import pandas as pd
import dataiku
from main_functions import *            # assumes versions_folder, compare_seg_folder, read_and_merge_files, construct_file_paths, etc.
from layout_components_style import *   # all your style helpers
import dash_bootstrap_components as dbc

# ---- Styles
card_style = get_card_style()
table_style = get_table_styles()
button_style = get_button_style()
enter_box_style = get_enter_box_style()
hidden_style = {'display': 'none'}
visible_style = {'display': 'block'}
tooltip_style = get_tooltip_style()
question_mark_style = get_question_mark_style()
header_h2_style = get_header_h2_style()
subheader_style = get_subheader_style()
main_style = get_main_style()
left_panel_style = get_left_panel_style()
right_panel_style = get_right_panel_style()
table_styles = get_table_styles()
div_message_style = get_div_message_style()
card_margin = {'margin': '3px 0', 'padding': '0'}

# ---- Safe loaders (no global metadata_df / no folder I/O at import time)
def safe_load_metadata_df():
    """
    Load the metadata dataset safely and normalize columns used by the UI.
    """
    ds = dataiku.Dataset("metadata_dataset")
    df = ds.get_dataframe()
    df = df.copy()
    # Compose version_name and place it first
    df['version_name'] = df['session_name'] + "_" + df["segmentation_version"].astype(str)
    first = df.pop('version_name')
    df.insert(0, 'version_name', first)
    # Keep session_name + segmentation_version out of the UI table
    return df.drop(columns=['session_name', 'segmentation_version'])

def safe_versions_folder():
    """
    Return the versions folder handle from main_functions (already initialized there).
    Avoids calling get_path() at import-time.
    """
    return versions_folder

# -------------------- UI --------------------

# Top-right refresh button (front-end reload only)
restart_button_div = html.Div([
    html.Button("REFRESH", id="clear-icon", style={**button_style, 'padding': '4px 10px'}),
    dbc.Tooltip(
        "Reset the interface to its default state. This won't delete any saved sessions.",
        target="clear-icon",
        placement="left",
        style=tooltip_style
    ),
    html.Div(id='refresh-trigger', style={'display': 'none'})
], style={
    'position': 'fixed',
    'top': '10px',
    'right': '10px',
    'zIndex': 999,
    'background-color': 'white',
    'border': '1px solid #ccc',
    'padding': '6px',
    'border-radius': '6px',
    'box-shadow': '0 2px 4px rgba(0,0,0,0.1)'
})

app.layout = html.Div([
    restart_button_div,

    html.Div([
        # Left panel
        html.Div([
            html.H2([
                'Select Sessions',
                html.Span('?', id='tooltip-select-sessions', style=question_mark_style)
            ], style=header_h2_style),
            dbc.Tooltip(
                "Choose two or more segmentation sessions to join side-by-side using account_id. "
                "This helps explore how different segmentations overlap — for example, identifying "
                "doctors who are both low prescribers and high digital engagers, or tracking how "
                "individuals move across segments over time through different versions of the same segmentation.",
                target="tooltip-select-sessions",
                placement="right",
                style=tooltip_style
            ),
            dcc.Input(
                id='search-bar',
                type='text',
                placeholder='Search sessions',
                style=enter_box_style
            ),
            html.Div(
                id='card-container',
                className='card-container',
                style={
                    'height': '900px',
                    'overflow-y': 'scroll',
                    'border': '1px solid #ddd',
                    'padding': '10px',
                    'background-color': 'white'
                }
            )
        ], style=left_panel_style),

        # Right panel
        html.Div([
            html.Div([
                html.Div(
                    id="joined-data-container",
                    children=[
                        html.H2([
                            'Joined Data',
                            html.Span('?', id='tooltip-joined-data', style=question_mark_style)
                        ], style=header_h2_style),
                        dbc.Tooltip(
                            "The table below shows the result of a right join on the first session you selected, "
                            "using account_id as the key. If no data appears, it likely means there were no common accounts "
                            "between the selected sessions.",
                            target="tooltip-joined-data",
                            placement="right",
                            style=tooltip_style
                        ),
                        dash_table.DataTable(
                            id='merged-data-table',
                            columns=[],
                            data=[],
                            filter_action='native',
                            page_size=16,
                            page_action='native',
                            page_current=0,
                            **table_styles
                        ),
                        html.H2([
                            'Data Processing',
                            html.Span('?', id='tooltip-data-process-comp', style=question_mark_style)
                        ], style=header_h2_style),
                        dbc.Tooltip(
                            "Once sessions are joined, you can name and export the resulting dataset to your Dataiku Flow "
                            "within compare_sessions folder. Use it to explore intersections between segmentation versions "
                            "or methods for deeper targeting and analytics in the Dataiku Dashboards.",
                            target="tooltip-data-process-comp",
                            placement="right",
                            style=tooltip_style
                        ),
                    ],
                    style=hidden_style
                ),
                html.Div("Select sessions to compare.", id="select-message", style=visible_style)
            ]),

            # Export widgets
            html.Div([
                html.P(
                    "Export the merged data to your Flow to identify cross-segment patterns — "
                    "such as users appearing in multiple strategic groups across different segmentation approaches.",
                    id='export-description',
                    style={**div_message_style, 'display': 'none', 'margin-top': '15px'}
                ),
                dcc.Input(
                    id='file-name-input',
                    type='text',
                    placeholder='Enter a file name',
                    style={**enter_box_style, 'margin-top': '10px', 'display': 'none'}
                ),
                html.Button('EXPORT JOINED DATA', id='export-button', style=button_style),
                html.Div(id='success-message', style={'margin-top': '20px', 'display': 'none'})
            ]),
        ], style=right_panel_style)
    ], style=main_style)
])

# -------------------- Callbacks --------------------

@app.callback(
    Output('card-container', 'children'),
    Input('search-bar', 'value')
)
def update_cards(search_value):
    """
    Rebuild the session list on each search change (fresh metadata each time).
    If you prefer caching, see note below about a dcc.Store.
    """
    try:
        df = safe_load_metadata_df()
        print(df.head(2))  # debug
    except Exception as e:
        return [html.Div(html.P(f"Could not load metadata: {e}", style=div_message_style))]

    required_cols = {'version_name', 'status', 'username', 'datetime', 'method', 'description'}
    missing = required_cols - set(df.columns)
    if missing:
        return [html.Div(html.P(f"Metadata missing columns: {sorted(missing)}", style=div_message_style))]

    df = df.copy()
    df['version_name'] = df['version_name'].fillna('')
    if search_value:
        mask = df['version_name'].str.contains(str(search_value), case=False, na=False)
        filtered_df = df[mask]
    else:
        filtered_df = df

    cards = [
        html.Div(
            children=[
                dcc.Checklist(
                    options=[{'label': '', 'value': row['version_name']}],
                    value=[],
                    id={'type': 'session-card', 'index': int(idx)},
                    style={'margin-bottom': '5px'}
                ),
                html.Label(
                    f"{row['version_name']} ({row['status']})",
                    style={'font-weight': 'bold', 'margin': '3px 0', 'padding': '0'}
                ),
                html.P(f"User: {row['username']}", style=card_margin),
                html.P(f"Date: {row['datetime']}", style=card_margin),
                html.P(f"Method: {row['method']}", style=card_margin),
            ],
            style=card_style,
            className='hover-effect',
            id={'type': 'card-div', 'index': int(idx)}
        )
        for idx, row in filtered_df.iterrows()
    ]

    tooltips = [
        dbc.Tooltip(
            f"Description: {row['description']}",
            target={'type': 'card-div', 'index': int(idx)},
            placement="right",
            style=tooltip_style
        )
        for idx, row in filtered_df.iterrows()
    ]

    return cards + tooltips


@app.callback(
    [Output('joined-data-container', 'style'),
     Output('select-message', 'style'),
     Output('merged-data-table', 'data'),
     Output('merged-data-table', 'columns'),
     Output('file-name-input', 'style'),
     Output('export-button', 'style'),
     Output('export-description', 'style')],
    Input({'type': 'session-card', 'index': dash.dependencies.ALL}, 'value')
)
def update_table_and_visibility(selected_sessions_values):
    """
    Build the joined table using the sessions checked in the left panel.
    Loads fresh metadata via safe_load_metadata_df() to resolve selected labels → files.
    """
    # Flatten selected values
    selected_sessions = [item for sublist in selected_sessions_values for item in sublist]

    if not selected_sessions:
        return hidden_style, visible_style, [], [], hidden_style, hidden_style, hidden_style

    # Load metadata fresh (to resolve any new versions immediately)
    try:
        df = safe_load_metadata_df()
    except Exception as e:
        # If metadata fails, show a friendly message in place of the table.
        msg = html.P(f"Could not load metadata to build comparison: {e}", style=div_message_style)
        return hidden_style, visible_style, [], [], hidden_style, hidden_style, hidden_style

    # Filter metadata to selected sessions
    selected_sessions_df = df[df['version_name'].isin(selected_sessions)]
    sessions_list = selected_sessions_df['version_name'].tolist()

    # Build paths and read/merge files
    vfolder = safe_versions_folder()
    sessions_list_paths = construct_file_paths(sessions_list)
    merged_data = read_and_merge_files(vfolder, sessions_list_paths, sessions_list)

    data = merged_data.to_dict('records')
    columns = [{"name": col, "id": col} for col in merged_data.columns]

    return (
        visible_style,                 # show table container
        hidden_style,                  # hide "Select sessions" message
        data, columns,
        {**enter_box_style, **visible_style},  # show filename input
        {**button_style, **visible_style},     # show export button
        {**div_message_style, 'display': 'block'}  # show export description
    )


@app.callback(
    Output('file-name-input', 'value'),
    Output('success-message', 'children'),
    Output('success-message', 'style'),
    Input('export-button', 'n_clicks'),
    State('merged-data-table', 'data'),
    State('file-name-input', 'value')
)
def export_data(n_clicks, data, file_name):
    # No click or no data: Do nothing
    if not n_clicks:
        return file_name, '', hidden_style
    
    # Check if data is empty
    if not data:
        return file_name, html.P("No data to export."),  {**div_message_style, 'display': 'block'}

    # Validate file name
    if not file_name or re.search(r'[\\/*?:"<>|]', file_name):
        return file_name, html.P("Invalid filename. Avoid special characters like \\ / : * ? \" < > |"),  {**div_message_style, 'display': 'block'}

    # Export to Dataiku folder
    df = pd.DataFrame(data)
    compare_filename = f"{file_name}.csv"
    compare_seg_folder.upload_data(compare_filename, df.to_csv(index=False).encode("utf-8"))

    # Success feedback
    return '', html.P("Data successfully exported to the Compare Sessions folder!"),  {**div_message_style, 'display': 'block'}



# ---- UI-only refresh (front-end reload)
@app.callback(
    Output('refresh-trigger', 'children'),
    Input('clear-icon', 'n_clicks'),
    prevent_initial_call=True
)
def trigger_clear(n_clicks):
    if n_clicks:
        return "REFRESH"
    return dash.no_update

app.clientside_callback(
    """
    function(trigger) {
        if (trigger === "REFRESH") {
            setTimeout(function() {
                window.location.reload();
            }, 500);
        }
        return '';
    }
    """,
    Output('refresh-trigger', 'style'),
    Input('refresh-trigger', 'children'),
    prevent_initial_call=True
)
